/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.server.security.enterprise.auth;

import com.google.common.jimfs.Configuration;
import com.google.common.jimfs.Jimfs;
import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Future;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.neo4j.graphdb.mockfs.DelegatingFileSystemAbstraction;
import org.neo4j.io.fs.DelegateFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.kernel.api.exceptions.InvalidArgumentsException;
import org.neo4j.logging.AssertableLogProvider;
import org.neo4j.logging.LogProvider;
import org.neo4j.logging.NullLogProvider;
import org.neo4j.server.security.auth.FileRepositorySerializer;
import org.neo4j.server.security.auth.ListSnapshot;
import org.neo4j.server.security.auth.exception.ConcurrentModificationException;
import org.neo4j.server.security.enterprise.auth.FileRoleRepository;
import org.neo4j.server.security.enterprise.auth.RoleRecord;
import org.neo4j.server.security.enterprise.auth.RoleRepository;
import org.neo4j.string.UTF8;
import org.neo4j.test.DoubleLatch;
import org.neo4j.test.rule.concurrent.ThreadingRule;

@RunWith(value=Parameterized.class)
public class FileRoleRepositoryTest {
    private File roleFile = new File("dbms", "roles");
    private LogProvider logProvider = NullLogProvider.getInstance();
    private FileSystemAbstraction fs;
    private RoleRepository roleRepository;
    @Rule
    public ExpectedException thrown = ExpectedException.none();
    @Rule
    public ThreadingRule threading = new ThreadingRule();

    @Parameterized.Parameters(name="{1} filesystem")
    public static Collection<Object[]> data() {
        return Arrays.asList({Configuration.unix(), "unix"}, {Configuration.osX(), "osX"}, {Configuration.windows(), "windows"});
    }

    public FileRoleRepositoryTest(Configuration fsConfig, String fsType) {
        this.fs = new DelegateFileSystemAbstraction(Jimfs.newFileSystem((Configuration)fsConfig));
    }

    @Before
    public void setup() {
        this.roleRepository = new FileRoleRepository(this.fs, this.roleFile, this.logProvider);
    }

    @After
    public void tearDown() throws IOException {
        this.fs.close();
    }

    @Test
    public void shouldStoreAndRetrieveRolesByName() throws Exception {
        RoleRecord role = new RoleRecord("admin", new String[]{"petra", "olivia"});
        this.roleRepository.create(role);
        RoleRecord result = this.roleRepository.getRoleByName(role.name());
        MatcherAssert.assertThat((Object)result, (Matcher)CoreMatchers.equalTo((Object)role));
    }

    @Test
    public void shouldPersistRoles() throws Throwable {
        RoleRecord role = new RoleRecord("admin", new String[]{"craig", "karl"});
        this.roleRepository.create(role);
        this.roleRepository = new FileRoleRepository(this.fs, this.roleFile, this.logProvider);
        this.roleRepository.start();
        RoleRecord resultByName = this.roleRepository.getRoleByName(role.name());
        MatcherAssert.assertThat((Object)resultByName, (Matcher)CoreMatchers.equalTo((Object)role));
    }

    @Test
    public void shouldNotFindRoleAfterDelete() throws Throwable {
        RoleRecord role = new RoleRecord("jake", new String[]{"admin"});
        this.roleRepository.create(role);
        this.roleRepository.delete(role);
        MatcherAssert.assertThat((Object)this.roleRepository.getRoleByName(role.name()), (Matcher)CoreMatchers.nullValue());
    }

    @Test
    public void shouldNotAllowComplexNames() throws Exception {
        this.roleRepository.assertValidRoleName("neo4j");
        this.roleRepository.assertValidRoleName("johnosbourne");
        this.roleRepository.assertValidRoleName("john_osbourne");
        org.neo4j.test.assertion.Assert.assertException(() -> this.roleRepository.assertValidRoleName(null), InvalidArgumentsException.class, (String)"The provided role name is empty.");
        org.neo4j.test.assertion.Assert.assertException(() -> this.roleRepository.assertValidRoleName(""), InvalidArgumentsException.class, (String)"The provided role name is empty.");
        org.neo4j.test.assertion.Assert.assertException(() -> this.roleRepository.assertValidRoleName(":"), InvalidArgumentsException.class, (String)"Role name ':' contains illegal characters. Use simple ascii characters and numbers.");
        org.neo4j.test.assertion.Assert.assertException(() -> this.roleRepository.assertValidRoleName("john osbourne"), InvalidArgumentsException.class, (String)"Role name 'john osbourne' contains illegal characters. Use simple ascii characters and numbers.");
        org.neo4j.test.assertion.Assert.assertException(() -> this.roleRepository.assertValidRoleName("john:osbourne"), InvalidArgumentsException.class, (String)"Role name 'john:osbourne' contains illegal characters. Use simple ascii characters and numbers.");
    }

    @Test
    public void shouldRecoverIfCrashedDuringMove() throws Throwable {
        final IOException exception = new IOException("simulated IO Exception on create");
        DelegatingFileSystemAbstraction craschingFileSystem = new DelegatingFileSystemAbstraction(this.fs){

            public void renameFile(File oldLocation, File newLocation, CopyOption ... copyOptions) throws IOException {
                if (FileRoleRepositoryTest.this.roleFile.getName().equals(newLocation.getName())) {
                    throw exception;
                }
                super.renameFile(oldLocation, newLocation, copyOptions);
            }
        };
        this.roleRepository = new FileRoleRepository((FileSystemAbstraction)craschingFileSystem, this.roleFile, this.logProvider);
        this.roleRepository.start();
        RoleRecord role = new RoleRecord("admin", new String[]{"jake"});
        try {
            this.roleRepository.create(role);
            Assert.fail((String)"Expected an IOException");
        }
        catch (IOException e) {
            Assert.assertSame((Object)exception, (Object)e);
        }
        Assert.assertFalse((boolean)craschingFileSystem.fileExists(this.roleFile));
        MatcherAssert.assertThat((Object)craschingFileSystem.listFiles(this.roleFile.getParentFile()).length, (Matcher)CoreMatchers.equalTo((Object)0));
    }

    @Test
    public void shouldThrowIfUpdateChangesName() throws Throwable {
        RoleRecord role = new RoleRecord("admin", new String[]{"steve", "bob"});
        this.roleRepository.create(role);
        RoleRecord updatedRole = new RoleRecord("admins", new String[]{"steve", "bob"});
        try {
            this.roleRepository.update(role, updatedRole);
            Assert.fail((String)"expected exception not thrown");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        MatcherAssert.assertThat((Object)this.roleRepository.getRoleByName(role.name()), (Matcher)CoreMatchers.equalTo((Object)role));
    }

    @Test
    public void shouldThrowIfExistingRoleDoesNotMatch() throws Throwable {
        RoleRecord role = new RoleRecord("admin", new String[]{"jake"});
        this.roleRepository.create(role);
        RoleRecord modifiedRole = new RoleRecord("admin", new String[]{"jake", "john"});
        RoleRecord updatedRole = new RoleRecord("admin", new String[]{"john"});
        try {
            this.roleRepository.update(modifiedRole, updatedRole);
            Assert.fail((String)"expected exception not thrown");
        }
        catch (ConcurrentModificationException concurrentModificationException) {
            // empty catch block
        }
    }

    @Test
    public void shouldFailOnReadingInvalidEntries() throws Throwable {
        AssertableLogProvider logProvider = new AssertableLogProvider();
        this.fs.mkdirs(this.roleFile.getParentFile());
        FileRepositorySerializer.writeToFile((FileSystemAbstraction)this.fs, (File)this.roleFile, (byte[])UTF8.encode((String)"neo4j:admin\nadmin:admin:\n"));
        this.roleRepository = new FileRoleRepository(this.fs, this.roleFile, (LogProvider)logProvider);
        this.thrown.expect(IllegalStateException.class);
        this.thrown.expectMessage(CoreMatchers.startsWith((String)"Failed to read role file '"));
        try {
            this.roleRepository.start();
        }
        catch (IllegalStateException e) {
            MatcherAssert.assertThat((Object)this.roleRepository.numberOfRoles(), (Matcher)CoreMatchers.equalTo((Object)0));
            logProvider.assertExactly(new AssertableLogProvider.LogMatcher[]{AssertableLogProvider.inLog(FileRoleRepository.class).error("Failed to read role file \"%s\" (%s)", new Object[]{this.roleFile.getAbsolutePath(), "wrong number of line fields [line 2]"})});
            throw e;
        }
    }

    @Test
    public void shouldNotAddEmptyUserToRole() throws Throwable {
        this.fs.mkdirs(this.roleFile.getParentFile());
        FileRepositorySerializer.writeToFile((FileSystemAbstraction)this.fs, (File)this.roleFile, (byte[])UTF8.encode((String)"admin:neo4j\nreader:\n"));
        this.roleRepository = new FileRoleRepository(this.fs, this.roleFile, this.logProvider);
        this.roleRepository.start();
        RoleRecord role = this.roleRepository.getRoleByName("admin");
        Assert.assertTrue((String)"neo4j should be assigned to 'admin'", (boolean)role.users().contains("neo4j"));
        Assert.assertTrue((String)"only one admin should exist", (role.users().size() == 1 ? 1 : 0) != 0);
        role = this.roleRepository.getRoleByName("reader");
        Assert.assertTrue((String)"no users should be assigned to 'reader'", (boolean)role.users().isEmpty());
    }

    @Test
    public void shouldProvideRolesByUsernameEvenIfMidSetRoles() throws Throwable {
        this.roleRepository = new FileRoleRepository(this.fs, this.roleFile, this.logProvider);
        this.roleRepository.create(new RoleRecord("admin", new String[]{"oskar"}));
        DoubleLatch latch = new DoubleLatch(2);
        Future setUsers = this.threading.execute(o -> {
            this.roleRepository.setRoles((ListSnapshot)new HangingListSnapshot(latch, 10L, Collections.emptyList()));
            return null;
        }, null);
        latch.startAndWaitForAllToStart();
        MatcherAssert.assertThat((Object)this.roleRepository.getRoleNamesByUsername("oskar"), (Matcher)Matchers.containsInAnyOrder((Object[])new String[]{"admin"}));
        latch.finish();
        setUsers.get();
    }

    class HangingListSnapshot
    extends ListSnapshot<RoleRecord> {
        private final DoubleLatch latch;

        HangingListSnapshot(DoubleLatch latch, long timestamp, List<RoleRecord> values) {
            super(timestamp, values, true);
            this.latch = latch;
        }

        public long timestamp() {
            this.latch.start();
            this.latch.finishAndWaitForAllToFinish();
            return super.timestamp();
        }
    }
}

