/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.token;

import java.util.Arrays;
import java.util.HashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.IntPredicate;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.neo4j.exceptions.KernelException;
import org.neo4j.token.CreatingTokenHolder;
import org.neo4j.token.TokenCreator;
import org.neo4j.token.api.NamedToken;
import org.neo4j.token.api.TokenHolder;
import org.neo4j.token.api.TokenNotFoundException;

class CreatingTokenHolderTest {
    private TokenCreator creator;
    private TokenHolder holder;

    CreatingTokenHolderTest() {
    }

    @BeforeEach
    void setUp() {
        this.creator = (TokenCreator)Mockito.mock(TokenCreator.class);
        this.holder = new CreatingTokenHolder(this.creator, "Dummy");
    }

    @Test
    void mustKnownItsTokenType() {
        org.junit.jupiter.api.Assertions.assertEquals((Object)"Dummy", (Object)this.holder.getTokenType());
    }

    @Test
    void mustCreateAndNotCacheNewTokens() throws Exception {
        Mockito.when((Object)this.creator.createToken("token", false)).thenReturn((Object)42);
        Assertions.assertThat((int)this.holder.getOrCreateId("token")).isEqualTo(42);
        Assertions.assertThat((int)this.holder.getIdByName("token")).isEqualTo(-1);
        this.holder.addToken(new NamedToken("token", 42));
        Assertions.assertThat((int)this.holder.getOrCreateId("token")).isEqualTo(42);
        ((TokenCreator)Mockito.verify((Object)this.creator)).createToken("token", false);
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.creator});
    }

    @Test
    void mustBatchCreateAndCacheNewTokens() throws Exception {
        this.mockAssignNewTokenIdsInBatch(new AtomicInteger(42));
        String[] names = new String[]{"token"};
        int[] ids = new int[1];
        this.holder.getOrCreateIds(names, ids);
        Assertions.assertThat((int)ids[0]).isEqualTo(42);
        Assertions.assertThat((int)this.holder.getIdByName("token")).isEqualTo(-1);
        this.holder.addToken(new NamedToken("token", 42));
        this.holder.getOrCreateIds(names, ids);
        Assertions.assertThat((int)ids[0]).isEqualTo(42);
        ((TokenCreator)Mockito.verify((Object)this.creator)).createTokens((String[])ArgumentMatchers.any(String[].class), (int[])ArgumentMatchers.any(int[].class), ArgumentMatchers.eq((boolean)false), (IntPredicate)ArgumentMatchers.any(IntPredicate.class));
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.creator});
    }

    @Test
    void mustBatchCreateAndCacheNewInternalTokens() throws Exception {
        this.mockAssignNewTokenIdsInBatch(new AtomicInteger(42));
        String[] names = new String[]{"token"};
        int[] ids = new int[1];
        this.holder.getOrCreateInternalIds(names, ids);
        Assertions.assertThat((int)ids[0]).isEqualTo(42);
        org.junit.jupiter.api.Assertions.assertThrows(TokenNotFoundException.class, () -> this.holder.getInternalTokenById(42));
        this.holder.addToken(new NamedToken("token", 42, true));
        this.holder.getOrCreateInternalIds(names, ids);
        Assertions.assertThat((int)ids[0]).isEqualTo(42);
        ((TokenCreator)Mockito.verify((Object)this.creator)).createTokens((String[])ArgumentMatchers.any(String[].class), (int[])ArgumentMatchers.any(int[].class), ArgumentMatchers.eq((boolean)true), (IntPredicate)ArgumentMatchers.any(IntPredicate.class));
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.creator});
    }

    @Test
    void batchTokenGetMustReturnWhetherThereWereUnresolvedTokens() {
        this.holder.setInitialTokens(Arrays.asList(CreatingTokenHolderTest.token("a", 1), CreatingTokenHolderTest.token("b", 2)));
        String[] names = new String[]{"a", "X", "b"};
        int[] ids = new int[]{-1, -1, -1};
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.holder.getIdsByNames(names, ids));
        Assertions.assertThat((int)ids[0]).isEqualTo(1);
        Assertions.assertThat((int)ids[1]).isEqualTo(-1);
        Assertions.assertThat((int)ids[2]).isEqualTo(2);
        names = new String[]{"a", "b"};
        ids = new int[]{-1, -1};
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.holder.getIdsByNames(names, ids));
        Assertions.assertThat((int)ids[0]).isEqualTo(1);
        Assertions.assertThat((int)ids[1]).isEqualTo(2);
    }

    @Test
    void batchTokenCreateMustIgnoreExistingTokens() throws Exception {
        this.initialTokensABC();
        AtomicInteger nextId = new AtomicInteger(42);
        this.mockAssignNewTokenIdsInBatch(nextId);
        String[] names = new String[]{"b", "X", "a", "Y", "c"};
        int[] ids = new int[names.length];
        this.holder.getOrCreateIds(names, ids);
        Assertions.assertThat((int)ids.length).isEqualTo(5);
        Assertions.assertThat((int)ids[0]).isEqualTo(2);
        Assertions.assertThat((int)ids[1]).isIn(new Object[]{42, 43});
        Assertions.assertThat((int)ids[2]).isEqualTo(1);
        Assertions.assertThat((int)ids[3]).isIn(new Object[]{42, 43});
        Assertions.assertThat((int)ids[4]).isEqualTo(3);
        Assertions.assertThat((int)nextId.get()).isEqualTo(44);
        org.junit.jupiter.api.Assertions.assertThrows(TokenNotFoundException.class, () -> this.holder.getTokenById(42));
        org.junit.jupiter.api.Assertions.assertThrows(TokenNotFoundException.class, () -> this.holder.getTokenById(43));
    }

    @Test
    void batchTokenCreateInternalMustIgnoreExistingTokens() throws Exception {
        this.initialInternalTokensABC();
        AtomicInteger nextId = new AtomicInteger(42);
        this.mockAssignNewTokenIdsInBatch(nextId);
        String[] names = new String[]{"b", "X", "a", "Y", "c"};
        int[] ids = new int[names.length];
        this.holder.getOrCreateInternalIds(names, ids);
        Assertions.assertThat((int)ids.length).isEqualTo(5);
        Assertions.assertThat((int)ids[0]).isEqualTo(2);
        Assertions.assertThat((int)ids[1]).isIn(new Object[]{42, 43});
        Assertions.assertThat((int)ids[2]).isEqualTo(1);
        Assertions.assertThat((int)ids[3]).isIn(new Object[]{42, 43});
        Assertions.assertThat((int)ids[4]).isEqualTo(3);
        Assertions.assertThat((int)nextId.get()).isEqualTo(44);
        org.junit.jupiter.api.Assertions.assertThrows(TokenNotFoundException.class, () -> this.holder.getInternalTokenById(42));
        org.junit.jupiter.api.Assertions.assertThrows(TokenNotFoundException.class, () -> this.holder.getInternalTokenById(43));
    }

    @Test
    void batchTokenCreateMustNotConfusePublicAndInternalTokens() throws KernelException {
        this.mockAssignNewTokenIdsInBatch(new AtomicInteger(10));
        int[] ids = new int[2];
        this.holder.getOrCreateIds(new String[]{"a", "b"}, ids);
        Assertions.assertThat((int)ids[0]).isEqualTo(10);
        Assertions.assertThat((int)ids[1]).isEqualTo(11);
        this.holder.addToken(new NamedToken("a", 10));
        this.holder.addToken(new NamedToken("b", 11));
        this.holder.getOrCreateInternalIds(new String[]{"b", "c"}, ids);
        Assertions.assertThat((int)ids[0]).isEqualTo(12);
        Assertions.assertThat((int)ids[1]).isEqualTo(13);
        this.holder.addToken(new NamedToken("b", 12, true));
        this.holder.addToken(new NamedToken("c", 13, true));
        this.holder.getOrCreateIds(new String[]{"b", "c"}, ids);
        Assertions.assertThat((int)ids[0]).isEqualTo(11);
        Assertions.assertThat((int)ids[1]).isEqualTo(14);
        this.holder.addToken(new NamedToken("c", 14));
        this.holder.getOrCreateInternalIds(new String[]{"c", "d"}, ids);
        Assertions.assertThat((int)ids[0]).isEqualTo(13);
        Assertions.assertThat((int)ids[1]).isEqualTo(15);
    }

    private void mockAssignNewTokenIdsInBatch(AtomicInteger nextId) throws KernelException {
        ((TokenCreator)Mockito.doAnswer(inv -> {
            int[] ids = (int[])inv.getArgument(1);
            IntPredicate filter = (IntPredicate)inv.getArgument(3);
            for (int i = 0; i < ids.length; ++i) {
                if (!filter.test(i)) continue;
                ids[i] = nextId.getAndIncrement();
            }
            return null;
        }).when((Object)this.creator)).createTokens((String[])ArgumentMatchers.any(String[].class), (int[])ArgumentMatchers.any(int[].class), ((Boolean)ArgumentMatchers.any(Boolean.TYPE)).booleanValue(), (IntPredicate)ArgumentMatchers.any(IntPredicate.class));
    }

    private void initialTokensABC() throws KernelException {
        this.holder.setInitialTokens(Arrays.asList(CreatingTokenHolderTest.token("a", 1), CreatingTokenHolderTest.token("b", 2)));
        Mockito.when((Object)this.creator.createToken("c", false)).thenReturn((Object)3);
        Assertions.assertThat((int)this.holder.getOrCreateId("c")).isEqualTo(3);
        this.holder.addToken(new NamedToken("c", 3));
    }

    private void initialInternalTokensABC() {
        this.holder.setInitialTokens(Arrays.asList(CreatingTokenHolderTest.token("a", 1, true), CreatingTokenHolderTest.token("b", 2, true), CreatingTokenHolderTest.token("c", 3, true)));
    }

    @Test
    void batchTokenCreateMustDeduplicateTokenCreates() throws Exception {
        this.initialTokensABC();
        AtomicInteger nextId = new AtomicInteger(42);
        this.mockAssignNewTokenIdsInBatch(nextId);
        String[] names = new String[]{"b", "b", "X", "a", "X", "c"};
        int[] ids = new int[names.length];
        this.holder.getOrCreateIds(names, ids);
        Assertions.assertThat((int)ids.length).isEqualTo(6);
        Assertions.assertThat((int)ids[0]).isEqualTo(2);
        Assertions.assertThat((int)ids[1]).isEqualTo(2);
        Assertions.assertThat((int)ids[2]).isEqualTo(42);
        Assertions.assertThat((int)ids[3]).isEqualTo(1);
        Assertions.assertThat((int)ids[4]).isEqualTo(42);
        Assertions.assertThat((int)ids[5]).isEqualTo(3);
        Assertions.assertThat((int)nextId.get()).isEqualTo(43);
        org.junit.jupiter.api.Assertions.assertThrows(TokenNotFoundException.class, () -> this.holder.getTokenById(42));
    }

    @Test
    void batchTokenCreateMustThrowOnArraysOfDifferentLengths() {
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.holder.getOrCreateIds(new String[3], new int[2]));
    }

    @Test
    void shouldClearTokensAsPartOfInitialTokenLoading() {
        this.holder.setInitialTokens(Arrays.asList(CreatingTokenHolderTest.token("one", 1), CreatingTokenHolderTest.token("two", 2)));
        CreatingTokenHolderTest.assertTokens(this.holder.getAllTokens(), CreatingTokenHolderTest.token("one", 1), CreatingTokenHolderTest.token("two", 2));
        this.holder.setInitialTokens(Arrays.asList(CreatingTokenHolderTest.token("two", 2), CreatingTokenHolderTest.token("three", 3), CreatingTokenHolderTest.token("four", 4)));
        CreatingTokenHolderTest.assertTokens(this.holder.getAllTokens(), CreatingTokenHolderTest.token("two", 2), CreatingTokenHolderTest.token("three", 3), CreatingTokenHolderTest.token("four", 4));
    }

    private static void assertTokens(Iterable<NamedToken> allTokens, NamedToken ... expectedTokens) {
        HashMap<String, NamedToken> existing = new HashMap<String, NamedToken>();
        for (NamedToken token : allTokens) {
            existing.put(token.name(), token);
        }
        HashMap<String, NamedToken> expected = new HashMap<String, NamedToken>();
        for (NamedToken token : expectedTokens) {
            expected.put(token.name(), token);
        }
        org.junit.jupiter.api.Assertions.assertEquals(expected, existing);
    }

    private static NamedToken token(String name, int id) {
        return CreatingTokenHolderTest.token(name, id, false);
    }

    private static NamedToken token(String name, int id, boolean internal) {
        return new NamedToken(name, id, internal);
    }
}

