/*
 * *##% 
 * JRedmine client
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.jredmine;

import java.io.IOException;
import org.nuiton.io.rest.RestException;
import org.nuiton.jredmine.model.Attachment;
import org.nuiton.jredmine.model.News;
import org.nuiton.jredmine.model.Version;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.io.rest.RestClient;
import org.nuiton.io.rest.RestRequest;
import java.io.InputStream;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;
import org.nuiton.io.rest.RestClientConfiguration;
import org.nuiton.jredmine.model.Issue;
import org.nuiton.jredmine.model.IssueCategory;
import org.nuiton.jredmine.model.IssuePriority;
import org.nuiton.jredmine.model.IssueStatus;
import org.nuiton.jredmine.model.Project;
import org.nuiton.jredmine.model.Tracker;
import org.nuiton.jredmine.model.User;
import org.nuiton.jredmine.model.io.xpp3.RedmineXpp3Helper;
import org.nuiton.jredmine.rest.RedmineRestClient;

/**
 * Default {@link RedmineService} implementation base on a {@link RestClient}
 * 
 * @author chemit
 * @since 1.0.0
 *
 * @plexus.component role="org.nuiton.jredmine.RedmineService"
 *                   role-hint="default"
 */
public class DefaultRedmineService implements RedmineService, RedmineServiceImplementor {

    protected static final Log log = LogFactory.getLog(DefaultRedmineService.class);
    /**
     * client Rest
     * @plexus.requirement role="org.nuiton.io.rest.RestClient"
     *                     role-hint="redmine"
     */
    protected RestClient session;
    /**
     * xpp3 xpp3Helper to transform xml stream to pojo
     */
    protected final RedmineXpp3Helper xpp3Helper;
    /**
     * internal state to known if service was init
     */
    protected boolean init;

    public DefaultRedmineService() {
        this.xpp3Helper = new RedmineXpp3Helper();
        if (log.isDebugEnabled()) {
            log.debug("new " + this);
        }
    }

    ///////////////////////////////////////////////////////////////////////////
    /// RedmineService implementation
    ///////////////////////////////////////////////////////////////////////////
    @Override
    public Version addVersion(String projectId, Version version) throws RedmineServiceException {
        checkInit();
        RestRequest r = getRequest(RedmineRestClient.ADD_VERSION, projectId, version);

        // send data and obtain created version
        try {

            InputStream stream = session.sendData(r);
            Version v = getDataFromStream(Version.class, stream);
            return v;
        } catch (Exception e) {
            throw new RedmineServiceException("could not send for reason " + e.getMessage(), e);
        }
    }

    @Override
    public Attachment addAttachment(String projectId, String versionId, Attachment attachement) throws RedmineServiceException {
        checkInit();
        RestRequest r = getRequest(RedmineRestClient.ADD_ATTACHMENT, projectId, versionId, attachement);

        // send data and obtain created attachment
        try {

            InputStream stream = session.sendData(r);
            Attachment v = getDataFromStream(Attachment.class, stream);
            return v;
        } catch (Exception e) {
            throw new RedmineServiceException("could not send for reason " + e.getMessage(), e);
        }
    }

    @Override
    public News addNews(String projectId, News news) throws RedmineServiceException {
        checkInit();
        RestRequest r = getRequest(RedmineRestClient.ADD_NEWS, projectId, news);

        // send data and obtain created news
        try {

            InputStream stream = session.sendData(r);
            News v = getDataFromStream(News.class, stream);
            return v;
        } catch (Exception e) {
            throw new RedmineServiceException("could not send for reason " + e.getMessage(), e);
        }
    }

    @Override
    public Version updateVersion(String projectId, Version version) throws RedmineServiceException {
        checkInit();
        RestRequest r = getRequest(RedmineRestClient.UPDATE_VERSION, projectId, version);

        // send data and obtain updated version
        try {

            InputStream stream = session.sendData(r);
            Version v = getDataFromStream(Version.class, stream);
            return v;
        } catch (Exception e) {
            throw new RedmineServiceException("could not send for reason " + e.getMessage(), e);
        }
    }

    @Override
    public Version nextVersion(String projectId, String oldVersionId, Version newVersion) throws RedmineServiceException {
        checkInit();
        RestRequest r = getRequest(RedmineRestClient.NEXT_VERSION, projectId, oldVersionId, newVersion);
        // send data and obtain updated version
        try {

            InputStream stream = session.sendData(r);
            Version v = getDataFromStream(Version.class, stream);
            return v;
        } catch (Exception e) {
            throw new RedmineServiceException("could not send for reason " + e.getMessage(), e);
        }
    }

    @Override
    public IssueStatus[] getIssueStatuses() throws RedmineServiceException {
        return getDatas(IssueStatus.class);
    }

    @Override
    public IssuePriority[] getIssuePriorities() throws RedmineServiceException {
        return getDatas(IssuePriority.class);
    }

    @Override
    public Project[] getProjects() throws RedmineServiceException {
        return getDatas(Project.class);
    }

    @Override
    public IssueCategory[] getIssueCategories(String projectId) throws RedmineServiceException {
        return getDatas(IssueCategory.class, projectId);
    }

    @Override
    public Project getProject(String projectId) throws RedmineServiceException {
        return getData(Project.class, projectId);
    }

    @Override
    public Tracker[] getTrackers(String projectId) throws RedmineServiceException {
        return getDatas(Tracker.class, projectId);
    }

    @Override
    public News[] getNews(String projectId) throws RedmineServiceException {
        return getDatas(News.class, projectId);
    }

    @Override
    public User[] getUsers(String projectId) throws RedmineServiceException {
        return getDatas(User.class, projectId);
    }

    @Override
    public Version[] getVersions(String projectId) throws RedmineServiceException {
        return getDatas(Version.class, projectId);
    }

    @Override
    public Version getVersion(String projectId, String versionId) throws RedmineServiceException {
        return getData(Version.class, projectId, versionId);
    }

    @Override
    public Attachment[] getAttachments(String projectId, String versionId) throws RedmineServiceException {
        return getDatas(Attachment.class, projectId, versionId);
    }

    @Override
    public Issue[] getIssues(String projectId, String versionId) throws RedmineServiceException {
        return getDatas(Issue.class, projectId, versionId);
    }

    ///////////////////////////////////////////////////////////////////////////
    /// RedmineServiceImplementor implementation
    ///////////////////////////////////////////////////////////////////////////
    @Override
    public boolean isInit() {
        return init;
    }

    @Override
    public RedmineServiceImplementor init(RestClientConfiguration configuration) throws RedmineServiceException {
        session.setConfiguration(configuration);
        RedmineServiceImplementor result = init(session);
        return result;
    }

    @Override
    public RedmineServiceImplementor init(RestClient session) throws RedmineServiceException {
        if (init) {
            throw new IllegalStateException("the client " + this + " was already init!");
        }
        try {
            this.session = session;

            if (!session.isOpen()) {
                session.open();
            }

            init = true;
        } catch (Exception e) {
            throw new RedmineServiceException("could not init service for reason " + e.getMessage(), e);
        }
        return this;
    }

    @Override
    public void destroy() throws RedmineServiceException {
        checkInit();
        try {
            if (session.isOpen()) {
                try {
                    session.close();
                } catch (RestException ex) {
                    throw new RedmineServiceException("has problem while closing Rest client " + ex.getMessage(), ex);
                }
            }
        } finally {
            if (session != null) {
                session = null;
            }
            init = false;
        }
    }

    @Override
    public <T> T getData(Class<T> type, Object... args) throws RedmineServiceException {
        checkInit();

        InputStream stream = getDataStream(type, args);

        if (stream == null) {
            return null;
        }

        T result = getDataFromStream(type, stream);
        return result;
    }

    @Override
    public <T> T[] getDatas(Class<T> type, Object... args) throws RedmineServiceException {
        checkInit();

        InputStream stream = getDatasStream(type, args);

        if (stream == null) {
            return null;
        }

        T[] result = getDatasFromStream(type, stream);
        return result;
    }

    @Override
    public <T> InputStream getDataStream(Class<T> type, Object... args) throws RedmineServiceException {

        InputStream stream = getDataStream(type, false, args);
        return stream;
    }

    @Override
    public <T> InputStream getDatasStream(Class<T> type, Object... args) throws RedmineServiceException {

        InputStream stream = getDataStream(type, true, args);
        return stream;
    }

    ///////////////////////////////////////////////////////////////////////////
    /// Internal methods
    ///////////////////////////////////////////////////////////////////////////
    protected <T> InputStream getDataStream(Class<T> type, boolean array, Object... args) throws RedmineServiceException {

        String requestName = type.getName() + (array ? "list" : "detail");
        RestRequest r = getRequest(requestName, args);

        // obtain data from rest client
        try {

            InputStream stream = session.askData(r);
            return stream;
        } catch (Exception e) {
            throw new RedmineServiceException("could not botain data stream for type " + type + " for reason " + e.getMessage(), e);
        }
    }

    protected RestRequest getRequest(String requestName, Object... args) throws RedmineServiceException {
        RestRequest r;
        try {
            r = session.getRequest(requestName, args);
        } catch (Exception e) {
            throw new RedmineServiceException("could not find the request named " + requestName + " for reason " + e.getMessage(), e);
        }
        if (r == null) {
            throw new RedmineServiceException("could not find the request named " + requestName);
        }
        return r;
    }

    protected <T> T getDataFromStream(Class<T> type, InputStream stream) throws RedmineServiceException {

        try {
            T result = xpp3Helper.readObject(type, stream);
            return result;
        } catch (IOException ex) {
            throw new RedmineServiceException("could not obtain datas of type " + type + " for reason " + ex.getMessage(), ex);
        } catch (XmlPullParserException ex) {
            throw new RedmineServiceException("could not obtain datas of type " + type + " for reason " + ex.getMessage(), ex);
        }
    }

    protected <T> T[] getDatasFromStream(Class<T> type, InputStream stream) throws RedmineServiceException {

        try {
            T[] result = xpp3Helper.readObjects(type, stream);
            return result;
        } catch (IOException ex) {
            throw new RedmineServiceException("could not obtain datas of type " + type + " for reason " + ex.getMessage(), ex);
        } catch (XmlPullParserException ex) {
            throw new RedmineServiceException("could not obtain datas of type " + type + " for reason " + ex.getMessage(), ex);
        }
    }

    protected void checkInit() throws IllegalStateException {
        if (!init) {
            throw new IllegalStateException("the client " + this + " is not init!");
        }
    }
}
