/*
 * *##% 
 * JRedmine client
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.jredmine;

import org.nuiton.jredmine.model.Attachment;
import org.nuiton.jredmine.model.Issue;
import org.nuiton.jredmine.model.IssueCategory;
import org.nuiton.jredmine.model.IssuePriority;
import org.nuiton.jredmine.model.IssueStatus;
import org.nuiton.jredmine.model.News;
import org.nuiton.jredmine.model.Project;
import org.nuiton.jredmine.model.Tracker;
import org.nuiton.jredmine.model.User;
import org.nuiton.jredmine.model.Version;

/**
 * The service that mirrors the rails {@code jredmine} plugin.
 *
 * @author chemit
 * @since 1.0.0
 */
public interface RedmineService {

    /** Plexus lookup name */
    String ROLE = RedmineService.class.getName();

    /**
     * Obtain all accessible projets for the loggued user.
     * 
     * @return all the projects
     * @throws RedmineServiceException if any pb
     * @see Project
     */
    Project[] getProjects() throws RedmineServiceException;

    /**
     * Obtain all the priorities defined on a {@link Issue}.
     *
     * <b>Note : </b> The priorities are common for all projects.
     *
     * @return all the issue properties
     * @throws RedmineServiceException
     * @see IssuePriority
     */
    IssuePriority[] getIssuePriorities() throws RedmineServiceException;

    /**
     * Obtain all the statuses defined on a {@link Issue}.
     *
     * <b>Note : </b> The statuses are common for all projects.
     *
     * @return all the issue statuses
     * @throws RedmineServiceException
     * @see IssueStatus
     */
    IssueStatus[] getIssueStatuses() throws RedmineServiceException;

    /**
     * Obtain a project given his name.
     * 
     * @param projectId the name of the project
     * @return the project 
     * @throws RedmineServiceException
     * @see Project
     */
    Project getProject(String projectId) throws RedmineServiceException;

    /**
     * Obtain all categories defined on issues for a given project.
     * 
     * @param projectId the name of the project
     * @return the categories of issues for the given project.
     * @throws RedmineServiceException
     * @see IssueCategory
     */
    IssueCategory[] getIssueCategories(String projectId) throws RedmineServiceException;

    /**
     * Obtain all trackers defined on a given project.
     *
     * @param projectId the name of the project
     * @return the trackers for the given project.
     * @throws RedmineServiceException
     * @see Tracker
     */
    Tracker[] getTrackers(String projectId) throws RedmineServiceException;

    /**
     * Obtain all news defined on a given project.
     *
     * @param projectId the name of the project
     * @return the news for the given project.
     * @throws RedmineServiceException
     * @see News
     */
    News[] getNews(String projectId) throws RedmineServiceException;

    /**
     * Obtain all users defined on a given project.
     *
     * @param projectId the name of the project
     * @return the users for the given project.
     * @throws RedmineServiceException
     * @see User
     */
    User[] getUsers(String projectId) throws RedmineServiceException;

    /**
     * Obtain all versions defined on a given project.
     *
     * @param projectId the name of the project
     * @return the versions of the given project.
     * @throws RedmineServiceException
     * @see Version
     */
    Version[] getVersions(String projectId) throws RedmineServiceException;

    /**
     * Obtain a specific version for a given project.
     *
     * @param projectId the name of the project
     * @param versionId the name of the version
     * @return the version
     * @throws RedmineServiceException
     * @see Version
     */
    Version getVersion(String projectId, String versionId) throws RedmineServiceException;

    /**
     * Obtain all issues for a specific version on a given project
     *
     * @param projectId the name of the project
     * @param versionId the name of the version
     * @return the issues
     * @throws RedmineServiceException
     * @see Issue
     */
    Issue[] getIssues(String projectId, String versionId) throws RedmineServiceException;

    /**
     * Obtain all attachments for a specific version on a given project
     *
     * @param projectId the name of the project
     * @param versionId the name of the version
     * @return the attachments
     * @throws RedmineServiceException
     * @see Attachment
     */
    Attachment[] getAttachments(String projectId, String versionId) throws RedmineServiceException;

    /**
     * Add a version for a given project.
     * 
     * @param projectId the name of the project
     * @param version the version to add
     * @return the added version
     * @throws RedmineServiceException
     * @see Version
     */
    Version addVersion(String projectId, Version version) throws RedmineServiceException;

    /**
     * Update a version for a given project.
     *
     * @param projectId the name of the project
     * @param version the version to update
     * @return the updated version
     * @throws RedmineServiceException
     * @see Version
     */
    Version updateVersion(String projectId, Version version) throws RedmineServiceException;

    /**
     * Prepare a new version (create it or update it).
     *
     * If the {@code oldVersionName} is given, then all issues unclosed from this
     * old version will be move to the new version.
     *
     * @param projectId the name of the project
     * @param oldVersionName the name of the old version (optional)
     * @param newVersion the newVersion to create or update
     * @return the created version
     * @throws RedmineServiceException
     * @see Version
     */
    Version nextVersion(String projectId, String oldVersionName, Version newVersion) throws RedmineServiceException;

    /**
     * Add a news for a given project.
     *
     * @param projectId the name of the project
     * @param news the news to add
     * @return the added news.
     * @throws RedmineServiceException
     * @see News
     */
    News addNews(String projectId, News news) throws RedmineServiceException;

    /**
     * Add a attachment for a given version of a given project.
     * 
     * @param projectId the name of the project
     * @param versionId the name of the version
     * @param attachement the attachment to add
     * @return the added attachment
     * @throws RedmineServiceException
     * @see Attachment
     */
    Attachment addAttachment(String projectId, String versionId, Attachment attachement) throws RedmineServiceException;
}
