/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.onosproject.incubator.net.l2monitoring.soam.loss;

import java.time.Instant;
import java.util.Collection;

import org.onosproject.incubator.net.l2monitoring.soam.MilliPct;
import org.onosproject.incubator.net.l2monitoring.soam.SoamId;

/**
 * A model of Loss Measurement from ITU Y.1731 Chapter 8.2, MEF 17, MEF 36.1 and MEF 39.
 *
 * In this model Loss Measurements entries are returned as a collection in the
 * MepEntry. In this way Loss Measurements are created by calling on the
 * Create Loss Measurement function, passing in any arguments needed to
 * configure it. The Loss Measurement is a result and not configured or
 * persisted in ONOS, but instead is is passed on to some analytics system.
 */
public interface LossMeasurementEntry extends LossMeasurementCreate {
    /**
     * Ian id that uniquely identifies a scheduled measurement.
     * It is automatically generated by the server on creation of a new measurement
     * @return An LM id
     */
    SoamId lmId();

    /**
     * Contains the Frame Loss Ratio in the forward direction calculated by this MEP.
     * from the last received SOAM PDU.
     * The FLR value is a ratio that is expressed as a percent with a value of
     * 0 (ratio 0.00) through 100000 (ratio 1.00).
     * @return Units are in milli-percent, where 1 indicates 0.001 per-cent
     */
    MilliPct measuredForwardFlr();

    /**
     * Contains the Frame Loss Ratio in the backward direction calculated by this MEP.
     * from the last received SOAM PDU.
     * The FLR value is a ratio that is expressed as a percent with a value of
     * 0 (ratio 0.00) through 100000 (ratio 1.00).
     * @return Units are in milli-percent, where 1 indicates 0.001 per-cent
     */
    MilliPct measuredBackwardFlr();

    /**
     * The availability status (the outcome of the last availability indicator) in the forward direction.
     * based upon the last received SOAM PDU
     * @return enumerated availability value
     */
    AvailabilityType measuredAvailabilityForwardStatus();

    /**
     * The availability status (the outcome of the last availability indicator) in the backward direction.
     * based upon the last received SOAM PDU
     * @return enumerated availability value
     */
    AvailabilityType measuredAvailabilityBackwardStatus();

    /**
     * The time of the last transition between available and unavailable in the forward direction.
     * If there have been no transitions since the Loss Measurement Session was
     * started, this is set to 0
     * @return The transition time
     */
    Instant measuredForwardLastTransitionTime();

    /**
     * The time of the last transition between available and unavailable in the backward direction.
     * If there have been no transitions since the Loss Measurement Session was
     * started, this is set to 0
     * @return The transition time
     */
    Instant measuredBackwardLastTransitionTime();

    /**
     * The results for the current Measurement Interval in a SOAM Loss Measurement session.
     * gathered during the interval indicated by measurement-interval
     * @return An object with current measurements
     */
    LossMeasurementStatCurrent measurementCurrent();

    /**
     * The results for history Measurement Intervals in a SOAM Loss Measurement session.
     * @return An object with historic measurements
     */
    Collection<LossMeasurementStatHistory> measurementHistories();

    /**
     * The current results for a SOAM Loss Measurement session for availability statistics.
     * gathered during the interval indicated by availability-measurement-interval
     * @return An object with current availability
     */
    LossAvailabilityStatCurrent availabilityCurrent();

    /**
     * The results for availability history Measurement Intervals in a SOAM Loss Measurement session.
     * @return An object with historic availability
     */
    Collection<LossAvailabilityStatHistory> availabilityHistories();

    /**
     * Builder for {@link org.onosproject.incubator.net.l2monitoring.soam.loss.LossMeasurementEntry}.
     */
    public interface LmEntryBuilder extends LmCreateBuilder {
        LmEntryBuilder measuredForwardFlr(MilliPct measuredForwardFlr);

        LmEntryBuilder measuredBackwardFlr(MilliPct measuredBackwardFlr);

        LmEntryBuilder measuredAvailabilityForwardStatus(
                AvailabilityType measuredAvailabilityForwardStatus);

        LmEntryBuilder measuredAvailabilityBackwardStatus(
                AvailabilityType measuredAvailabilityBackwardStatus);

        LmEntryBuilder measuredForwardLastTransitionTime(
                Instant measuredForwardLastTransitionTime);

        LmEntryBuilder measuredBackwardLastTransitionTime(
                Instant measuredBackwardLastTransitionTime);

        LmEntryBuilder measurementCurrent(
                LossMeasurementStatCurrent measurementCurrent);

        LmEntryBuilder addToMeasurementHistories(
                LossMeasurementStatHistory history);

        LmEntryBuilder availabilityCurrent(
                LossAvailabilityStatCurrent availabilityCurrent);

        LmEntryBuilder addToAvailabilityHistories(LossAvailabilityStatHistory history);

        LossMeasurementEntry build();
    }

    /**
     * Options for Availability test types.
     */
    public enum AvailabilityType {
        AVAILABLE,
        UNAVAILABLE,
        UNKNOWN;
    }
}
