/*
 * Decompiled with CFR 0.152.
 */
package org.opencastproject.silencedetection.ffmpeg;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.opencastproject.mediapackage.MediaPackageException;
import org.opencastproject.mediapackage.Track;
import org.opencastproject.silencedetection.api.MediaSegment;
import org.opencastproject.silencedetection.api.MediaSegments;
import org.opencastproject.silencedetection.api.SilenceDetectionFailedException;
import org.opencastproject.util.NotFoundException;
import org.opencastproject.workspace.api.Workspace;
import org.osgi.framework.BundleContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FFmpegSilenceDetector {
    private static final Logger logger = LoggerFactory.getLogger(FFmpegSilenceDetector.class);
    public static final String FFMPEG_BINARY_CONFIG = "org.opencastproject.composer.ffmpeg.path";
    public static final String FFMPEG_BINARY_DEFAULT = "ffmpeg";
    private static final Long DEFAULT_SILENCE_MIN_LENGTH = 5000L;
    private static final Long DEFAULT_SILENCE_PRE_LENGTH = 2000L;
    private static final String DEFAULT_THRESHOLD_DB = "-40dB";
    private static final Long DEFAULT_VOICE_MIN_LENGTH = 60000L;
    private static String binary = "ffmpeg";
    private String filePath;
    private String trackId;
    private List<MediaSegment> segments = null;

    public static void init(BundleContext bundleContext) {
        String binaryPath = bundleContext.getProperty(FFMPEG_BINARY_CONFIG);
        try {
            if (StringUtils.isNotBlank((CharSequence)binaryPath)) {
                File binaryFile = new File(StringUtils.trim((String)binaryPath));
                if (binaryFile.exists()) {
                    binary = binaryFile.getAbsolutePath();
                } else {
                    logger.warn("FFmpeg binary file {} does not exist", (Object)StringUtils.trim((String)binaryPath));
                }
            }
        }
        catch (Exception ex) {
            logger.error("Failed to set ffmpeg binary path", (Throwable)ex);
        }
    }

    public FFmpegSilenceDetector(Properties properties, Track track, Workspace workspace) throws SilenceDetectionFailedException, MediaPackageException, IOException {
        if (null == properties) {
            properties = new Properties();
        }
        long minSilenceLength = this.parseLong(properties, "silence.min.length", DEFAULT_SILENCE_MIN_LENGTH);
        long minVoiceLength = this.parseLong(properties, "voice.min.length", DEFAULT_VOICE_MIN_LENGTH);
        long preSilenceLength = this.parseLong(properties, "silence.pre.length", DEFAULT_SILENCE_PRE_LENGTH);
        String thresholdDB = properties.getProperty("silence.threshold.db", DEFAULT_THRESHOLD_DB);
        this.trackId = track.getIdentifier();
        if (!track.hasAudio()) {
            logger.warn("Track {} has no audio stream to run a silece detection on", (Object)this.trackId);
            throw new SilenceDetectionFailedException("Element has no audio stream");
        }
        if (preSilenceLength > minSilenceLength) {
            logger.error("Pre silence length ({}) is configured to be greater than minimun silence length ({})", (Object)preSilenceLength, (Object)minSilenceLength);
            throw new SilenceDetectionFailedException("preSilenceLength > minSilenceLength");
        }
        try {
            File mediaFile = workspace.get(track.getURI());
            this.filePath = mediaFile.getAbsolutePath();
        }
        catch (NotFoundException e) {
            throw new SilenceDetectionFailedException("Error finding the media file in workspace", (Throwable)e);
        }
        catch (IOException e) {
            throw new SilenceDetectionFailedException("Error reading media file in workspace", (Throwable)e);
        }
        if (track.getDuration() == null) {
            throw new MediaPackageException("Track " + this.trackId + " does not have a duration");
        }
        logger.debug("Track {} loaded, duration is {} s", (Object)this.filePath, (Object)(track.getDuration() / 1000L));
        logger.info("Starting silence detection of {}", (Object)this.filePath);
        DecimalFormat decimalFmt = new DecimalFormat("0.000", new DecimalFormatSymbols(Locale.US));
        String minSilenceLengthInSeconds = decimalFmt.format((double)minSilenceLength / 1000.0);
        String filter = "silencedetect=noise=" + thresholdDB + ":duration=" + minSilenceLengthInSeconds;
        String[] command = new String[]{binary, "-nostats", "-nostdin", "-i", this.filePath, "-vn", "-filter:a", filter, "-f", "null", "-"};
        logger.info("Running {}", (Object)command);
        ProcessBuilder pbuilder = new ProcessBuilder(command);
        LinkedList<String> segmentsStrings = new LinkedList<String>();
        Process process = pbuilder.start();
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getErrorStream()));){
            String line = reader.readLine();
            while (null != line) {
                logger.debug("FFmpeg output: {}", (Object)line);
                if (line.startsWith("[silencedetect ")) {
                    segmentsStrings.add(line);
                }
                line = reader.readLine();
            }
        }
        catch (IOException e) {
            logger.error("Error executing ffmpeg", (Throwable)e);
        }
        LinkedList<MediaSegment> segmentsTmp = new LinkedList<MediaSegment>();
        if (segmentsStrings.size() == 0) {
            logger.info("No silence found. Adding one large segment.");
            segmentsTmp.add(new MediaSegment(0L, track.getDuration().longValue()));
        } else {
            long lastSilenceEnd = 0L;
            long lastSilenceStart = 0L;
            Pattern patternStart = Pattern.compile("silence_start\\:\\ \\d+\\.\\d+");
            Pattern patternEnd = Pattern.compile("silence_end\\:\\ \\d+\\.\\d+");
            for (String seginfo : segmentsStrings) {
                Matcher matcher = patternEnd.matcher(seginfo);
                String time = "";
                while (matcher.find()) {
                    time = matcher.group().substring(13);
                }
                if (!"".equals(time)) {
                    long silenceEnd = (long)(Double.parseDouble(time) * 1000.0);
                    if (silenceEnd <= lastSilenceEnd) continue;
                    logger.debug("Found silence end at {}", (Object)silenceEnd);
                    lastSilenceEnd = silenceEnd;
                    continue;
                }
                matcher = patternStart.matcher(seginfo);
                time = "";
                while (matcher.find()) {
                    time = matcher.group().substring(15);
                }
                if ("".equals(time)) continue;
                lastSilenceStart = (long)(Double.parseDouble(time) * 1000.0);
                logger.debug("Found silence start at {}", (Object)lastSilenceStart);
                if (lastSilenceStart - lastSilenceEnd <= minVoiceLength) continue;
                long segmentStart = Math.max(0L, lastSilenceEnd - preSilenceLength);
                logger.info("Adding segment from {} to {}", (Object)segmentStart, (Object)lastSilenceStart);
                segmentsTmp.add(new MediaSegment(segmentStart, lastSilenceStart));
            }
            if (lastSilenceStart < lastSilenceEnd && track.getDuration() - lastSilenceEnd > minVoiceLength) {
                long segmentStart = Math.max(0L, lastSilenceEnd - preSilenceLength);
                logger.info("Adding final segment from {} to {}", (Object)segmentStart, (Object)track.getDuration());
                segmentsTmp.add(new MediaSegment(segmentStart, track.getDuration().longValue()));
            }
        }
        logger.info("Segmentation of track {} yielded {} segments", (Object)this.trackId, (Object)segmentsTmp.size());
        this.segments = segmentsTmp;
    }

    private Long parseLong(Properties properties, String key, Long defaultValue) {
        try {
            return Long.parseLong(properties.getProperty(key, defaultValue.toString()));
        }
        catch (NumberFormatException e) {
            logger.warn("Configuration value for {} is invalid, using default value of {} instead", (Object)key, (Object)defaultValue);
            return defaultValue;
        }
    }

    public MediaSegments getMediaSegments() {
        if (this.segments == null) {
            return null;
        }
        return new MediaSegments(this.trackId, this.filePath, this.segments);
    }
}

