package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bmp.monitor.config.rev200120.odl.bmp.monitors.bmp.monitor.config;
import com.google.common.base.MoreObjects;
import java.lang.Boolean;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.inet.types.rev130715.IpAddressNoZone;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.inet.types.rev130715.PortNumber;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.controller.rfc2385.cfg.rev160324.Rfc2385Key;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link MonitoredRouterBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     MonitoredRouterBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new MonitoredRouterBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of MonitoredRouterBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see MonitoredRouterBuilder
 * @see Builder
 *
 */
public class MonitoredRouterBuilder implements Builder<MonitoredRouter> {

    private IpAddressNoZone _address;
    private Rfc2385Key _password;
    private PortNumber _port;
    private Boolean _active;
    private MonitoredRouterKey key;


    Map<Class<? extends Augmentation<MonitoredRouter>>, Augmentation<MonitoredRouter>> augmentation = Collections.emptyMap();

    public MonitoredRouterBuilder() {
    }

    public MonitoredRouterBuilder(MonitoredRouter base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<MonitoredRouter>>, Augmentation<MonitoredRouter>> aug =((AugmentationHolder<MonitoredRouter>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this.key = base.key();
        this._address = base.getAddress();
        this._password = base.getPassword();
        this._port = base.getPort();
        this._active = base.isActive();
    }


    public MonitoredRouterKey key() {
        return key;
    }
    
    public IpAddressNoZone getAddress() {
        return _address;
    }
    
    public Rfc2385Key getPassword() {
        return _password;
    }
    
    public PortNumber getPort() {
        return _port;
    }
    
    public Boolean isActive() {
        return _active;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<MonitoredRouter>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    public MonitoredRouterBuilder withKey(final MonitoredRouterKey key) {
        this.key = key;
        return this;
    }
    
    public MonitoredRouterBuilder setAddress(final IpAddressNoZone value) {
        this._address = value;
        return this;
    }
    
    public MonitoredRouterBuilder setPassword(final Rfc2385Key value) {
        this._password = value;
        return this;
    }
    
    public MonitoredRouterBuilder setPort(final PortNumber value) {
        this._port = value;
        return this;
    }
    
    public MonitoredRouterBuilder setActive(final Boolean value) {
        this._active = value;
        return this;
    }
    
    public MonitoredRouterBuilder addAugmentation(Class<? extends Augmentation<MonitoredRouter>> augmentationType, Augmentation<MonitoredRouter> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public MonitoredRouterBuilder removeAugmentation(Class<? extends Augmentation<MonitoredRouter>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public MonitoredRouter build() {
        return new MonitoredRouterImpl(this);
    }

    private static final class MonitoredRouterImpl
        extends AbstractAugmentable<MonitoredRouter>
        implements MonitoredRouter {
    
        private final IpAddressNoZone _address;
        private final Rfc2385Key _password;
        private final PortNumber _port;
        private final Boolean _active;
        private final MonitoredRouterKey key;
    
        MonitoredRouterImpl(MonitoredRouterBuilder base) {
            super(base.augmentation);
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new MonitoredRouterKey(base.getAddress());
            }
            this._address = key.getAddress();
            this._password = base.getPassword();
            this._port = base.getPort();
            this._active = base.isActive();
        }
    
        @Override
        public MonitoredRouterKey key() {
            return key;
        }
        
        @Override
        public IpAddressNoZone getAddress() {
            return _address;
        }
        
        @Override
        public Rfc2385Key getPassword() {
            return _password;
        }
        
        @Override
        public PortNumber getPort() {
            return _port;
        }
        
        @Override
        public Boolean isActive() {
            return _active;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_address);
            result = prime * result + Objects.hashCode(_password);
            result = prime * result + Objects.hashCode(_port);
            result = prime * result + Objects.hashCode(_active);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!MonitoredRouter.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            MonitoredRouter other = (MonitoredRouter)obj;
            if (!Objects.equals(_address, other.getAddress())) {
                return false;
            }
            if (!Objects.equals(_password, other.getPassword())) {
                return false;
            }
            if (!Objects.equals(_port, other.getPort())) {
                return false;
            }
            if (!Objects.equals(_active, other.isActive())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                MonitoredRouterImpl otherImpl = (MonitoredRouterImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<MonitoredRouter>>, Augmentation<MonitoredRouter>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("MonitoredRouter");
            CodeHelpers.appendValue(helper, "_address", _address);
            CodeHelpers.appendValue(helper, "_password", _password);
            CodeHelpers.appendValue(helper, "_port", _port);
            CodeHelpers.appendValue(helper, "_active", _active);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
