package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bmp.monitor.rev200120.afi.safi.route.counter;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.yang.types.rev130715.Gauge64;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.types.rev200120.AddressFamily;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.types.rev200120.SubsequentAddressFamily;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link AfiSafiBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     AfiSafiBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new AfiSafiBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of AfiSafiBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see AfiSafiBuilder
 * @see Builder
 *
 */
public class AfiSafiBuilder implements Builder<AfiSafi> {

    private Class<? extends AddressFamily> _afi;
    private Gauge64 _count;
    private Class<? extends SubsequentAddressFamily> _safi;
    private AfiSafiKey key;


    Map<Class<? extends Augmentation<AfiSafi>>, Augmentation<AfiSafi>> augmentation = Collections.emptyMap();

    public AfiSafiBuilder() {
    }
    public AfiSafiBuilder(org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.multiprotocol.rev180329.BgpTableType arg) {
        this._afi = arg.getAfi();
        this._safi = arg.getSafi();
    }

    public AfiSafiBuilder(AfiSafi base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<AfiSafi>>, Augmentation<AfiSafi>> aug =((AugmentationHolder<AfiSafi>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this.key = base.key();
        this._afi = base.getAfi();
        this._safi = base.getSafi();
        this._count = base.getCount();
    }

    /**
     * Set fields from given grouping argument. Valid argument is instance of one of following types:
     * <ul>
     * <li>org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.multiprotocol.rev180329.BgpTableType</li>
     * </ul>
     *
     * @param arg grouping object
     * @throws IllegalArgumentException if given argument is none of valid types
    */
    public void fieldsFrom(DataObject arg) {
        boolean isValidArg = false;
        if (arg instanceof org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.multiprotocol.rev180329.BgpTableType) {
            this._afi = ((org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.multiprotocol.rev180329.BgpTableType)arg).getAfi();
            this._safi = ((org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.multiprotocol.rev180329.BgpTableType)arg).getSafi();
            isValidArg = true;
        }
        CodeHelpers.validValue(isValidArg, arg, "[org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.multiprotocol.rev180329.BgpTableType]");
    }

    public AfiSafiKey key() {
        return key;
    }
    
    public Class<? extends AddressFamily> getAfi() {
        return _afi;
    }
    
    public Gauge64 getCount() {
        return _count;
    }
    
    public Class<? extends SubsequentAddressFamily> getSafi() {
        return _safi;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<AfiSafi>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    public AfiSafiBuilder withKey(final AfiSafiKey key) {
        this.key = key;
        return this;
    }
    
    public AfiSafiBuilder setAfi(final Class<? extends AddressFamily> value) {
        this._afi = value;
        return this;
    }
    
    public AfiSafiBuilder setCount(final Gauge64 value) {
        this._count = value;
        return this;
    }
    
    public AfiSafiBuilder setSafi(final Class<? extends SubsequentAddressFamily> value) {
        this._safi = value;
        return this;
    }
    
    public AfiSafiBuilder addAugmentation(Class<? extends Augmentation<AfiSafi>> augmentationType, Augmentation<AfiSafi> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public AfiSafiBuilder removeAugmentation(Class<? extends Augmentation<AfiSafi>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public AfiSafi build() {
        return new AfiSafiImpl(this);
    }

    private static final class AfiSafiImpl
        extends AbstractAugmentable<AfiSafi>
        implements AfiSafi {
    
        private final Class<? extends AddressFamily> _afi;
        private final Gauge64 _count;
        private final Class<? extends SubsequentAddressFamily> _safi;
        private final AfiSafiKey key;
    
        AfiSafiImpl(AfiSafiBuilder base) {
            super(base.augmentation);
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new AfiSafiKey(base.getAfi(), base.getSafi());
            }
            this._afi = key.getAfi();
            this._safi = key.getSafi();
            this._count = base.getCount();
        }
    
        @Override
        public AfiSafiKey key() {
            return key;
        }
        
        @Override
        public Class<? extends AddressFamily> getAfi() {
            return _afi;
        }
        
        @Override
        public Gauge64 getCount() {
            return _count;
        }
        
        @Override
        public Class<? extends SubsequentAddressFamily> getSafi() {
            return _safi;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_afi);
            result = prime * result + Objects.hashCode(_count);
            result = prime * result + Objects.hashCode(_safi);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!AfiSafi.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            AfiSafi other = (AfiSafi)obj;
            if (!Objects.equals(_afi, other.getAfi())) {
                return false;
            }
            if (!Objects.equals(_count, other.getCount())) {
                return false;
            }
            if (!Objects.equals(_safi, other.getSafi())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                AfiSafiImpl otherImpl = (AfiSafiImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<AfiSafi>>, Augmentation<AfiSafi>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("AfiSafi");
            CodeHelpers.appendValue(helper, "_afi", _afi);
            CodeHelpers.appendValue(helper, "_count", _count);
            CodeHelpers.appendValue(helper, "_safi", _safi);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
