package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.flowspec.rev200120.redirect.extended.community;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.types.rev200120.ShortAsNumber;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link RedirectExtendedCommunityBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     RedirectExtendedCommunityBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new RedirectExtendedCommunityBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of RedirectExtendedCommunityBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see RedirectExtendedCommunityBuilder
 * @see Builder
 *
 */
public class RedirectExtendedCommunityBuilder implements Builder<RedirectExtendedCommunity> {

    private ShortAsNumber _globalAdministrator;
    private byte[] _localAdministrator;


    Map<Class<? extends Augmentation<RedirectExtendedCommunity>>, Augmentation<RedirectExtendedCommunity>> augmentation = Collections.emptyMap();

    public RedirectExtendedCommunityBuilder() {
    }

    public RedirectExtendedCommunityBuilder(RedirectExtendedCommunity base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<RedirectExtendedCommunity>>, Augmentation<RedirectExtendedCommunity>> aug =((AugmentationHolder<RedirectExtendedCommunity>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._globalAdministrator = base.getGlobalAdministrator();
        this._localAdministrator = base.getLocalAdministrator();
    }


    public ShortAsNumber getGlobalAdministrator() {
        return _globalAdministrator;
    }
    
    public byte[] getLocalAdministrator() {
        return _localAdministrator == null ? null : _localAdministrator.clone();
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<RedirectExtendedCommunity>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    
    public RedirectExtendedCommunityBuilder setGlobalAdministrator(final ShortAsNumber value) {
        this._globalAdministrator = value;
        return this;
    }
    
    private static void check_localAdministratorLength(final byte[] value) {
        final int length = value.length;
        if (length == 4) {
            return;
        }
        CodeHelpers.throwInvalidLength("[[4..4]]", value);
    }
    
    public RedirectExtendedCommunityBuilder setLocalAdministrator(final byte[] value) {
        if (value != null) {
            check_localAdministratorLength(value);
            
        }
        this._localAdministrator = value;
        return this;
    }
    
    public RedirectExtendedCommunityBuilder addAugmentation(Class<? extends Augmentation<RedirectExtendedCommunity>> augmentationType, Augmentation<RedirectExtendedCommunity> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public RedirectExtendedCommunityBuilder removeAugmentation(Class<? extends Augmentation<RedirectExtendedCommunity>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public RedirectExtendedCommunity build() {
        return new RedirectExtendedCommunityImpl(this);
    }

    private static final class RedirectExtendedCommunityImpl
        extends AbstractAugmentable<RedirectExtendedCommunity>
        implements RedirectExtendedCommunity {
    
        private final ShortAsNumber _globalAdministrator;
        private final byte[] _localAdministrator;
    
        RedirectExtendedCommunityImpl(RedirectExtendedCommunityBuilder base) {
            super(base.augmentation);
            this._globalAdministrator = base.getGlobalAdministrator();
            this._localAdministrator = base.getLocalAdministrator();
        }
    
        @Override
        public ShortAsNumber getGlobalAdministrator() {
            return _globalAdministrator;
        }
        
        @Override
        public byte[] getLocalAdministrator() {
            return _localAdministrator == null ? null : _localAdministrator.clone();
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_globalAdministrator);
            result = prime * result + Arrays.hashCode(_localAdministrator);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!RedirectExtendedCommunity.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            RedirectExtendedCommunity other = (RedirectExtendedCommunity)obj;
            if (!Objects.equals(_globalAdministrator, other.getGlobalAdministrator())) {
                return false;
            }
            if (!Arrays.equals(_localAdministrator, other.getLocalAdministrator())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                RedirectExtendedCommunityImpl otherImpl = (RedirectExtendedCommunityImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<RedirectExtendedCommunity>>, Augmentation<RedirectExtendedCommunity>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("RedirectExtendedCommunity");
            CodeHelpers.appendValue(helper, "_globalAdministrator", _globalAdministrator);
            CodeHelpers.appendValue(helper, "_localAdministrator", _localAdministrator);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
