package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.flowspec.rev200120.redirect.ipv6.extended.community;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Integer;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.inet.types.rev130715.Ipv6AddressNoZone;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;
import org.opendaylight.yangtools.yang.common.Uint16;

/**
 * Class that builds {@link RedirectIpv6Builder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     RedirectIpv6Builder createTarget(int fooXyzzy, int barBaz) {
 *         return new RedirectIpv6BuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of RedirectIpv6Builder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see RedirectIpv6Builder
 * @see Builder
 *
 */
public class RedirectIpv6Builder implements Builder<RedirectIpv6> {

    private Ipv6AddressNoZone _globalAdministrator;
    private Uint16 _localAdministrator;


    Map<Class<? extends Augmentation<RedirectIpv6>>, Augmentation<RedirectIpv6>> augmentation = Collections.emptyMap();

    public RedirectIpv6Builder() {
    }

    public RedirectIpv6Builder(RedirectIpv6 base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<RedirectIpv6>>, Augmentation<RedirectIpv6>> aug =((AugmentationHolder<RedirectIpv6>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._globalAdministrator = base.getGlobalAdministrator();
        this._localAdministrator = base.getLocalAdministrator();
    }


    public Ipv6AddressNoZone getGlobalAdministrator() {
        return _globalAdministrator;
    }
    
    public Uint16 getLocalAdministrator() {
        return _localAdministrator;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<RedirectIpv6>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    
    public RedirectIpv6Builder setGlobalAdministrator(final Ipv6AddressNoZone value) {
        this._globalAdministrator = value;
        return this;
    }
    
    public RedirectIpv6Builder setLocalAdministrator(final Uint16 value) {
        this._localAdministrator = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setLocalAdministrator(Uint16)} instead.
     */
    @Deprecated(forRemoval = true)
    public RedirectIpv6Builder setLocalAdministrator(final Integer value) {
        return setLocalAdministrator(CodeHelpers.compatUint(value));
    }
    
    public RedirectIpv6Builder addAugmentation(Class<? extends Augmentation<RedirectIpv6>> augmentationType, Augmentation<RedirectIpv6> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public RedirectIpv6Builder removeAugmentation(Class<? extends Augmentation<RedirectIpv6>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public RedirectIpv6 build() {
        return new RedirectIpv6Impl(this);
    }

    private static final class RedirectIpv6Impl
        extends AbstractAugmentable<RedirectIpv6>
        implements RedirectIpv6 {
    
        private final Ipv6AddressNoZone _globalAdministrator;
        private final Uint16 _localAdministrator;
    
        RedirectIpv6Impl(RedirectIpv6Builder base) {
            super(base.augmentation);
            this._globalAdministrator = base.getGlobalAdministrator();
            this._localAdministrator = base.getLocalAdministrator();
        }
    
        @Override
        public Ipv6AddressNoZone getGlobalAdministrator() {
            return _globalAdministrator;
        }
        
        @Override
        public Uint16 getLocalAdministrator() {
            return _localAdministrator;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_globalAdministrator);
            result = prime * result + Objects.hashCode(_localAdministrator);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!RedirectIpv6.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            RedirectIpv6 other = (RedirectIpv6)obj;
            if (!Objects.equals(_globalAdministrator, other.getGlobalAdministrator())) {
                return false;
            }
            if (!Objects.equals(_localAdministrator, other.getLocalAdministrator())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                RedirectIpv6Impl otherImpl = (RedirectIpv6Impl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<RedirectIpv6>>, Augmentation<RedirectIpv6>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("RedirectIpv6");
            CodeHelpers.appendValue(helper, "_globalAdministrator", _globalAdministrator);
            CodeHelpers.appendValue(helper, "_localAdministrator", _localAdministrator);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
