package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.flowspec.rev200120.traffic.action.extended.community;
import com.google.common.base.MoreObjects;
import java.lang.Boolean;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link TrafficActionExtendedCommunityBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     TrafficActionExtendedCommunityBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new TrafficActionExtendedCommunityBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of TrafficActionExtendedCommunityBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see TrafficActionExtendedCommunityBuilder
 * @see Builder
 *
 */
public class TrafficActionExtendedCommunityBuilder implements Builder<TrafficActionExtendedCommunity> {

    private Boolean _sample;
    private Boolean _terminalAction;


    Map<Class<? extends Augmentation<TrafficActionExtendedCommunity>>, Augmentation<TrafficActionExtendedCommunity>> augmentation = Collections.emptyMap();

    public TrafficActionExtendedCommunityBuilder() {
    }

    public TrafficActionExtendedCommunityBuilder(TrafficActionExtendedCommunity base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<TrafficActionExtendedCommunity>>, Augmentation<TrafficActionExtendedCommunity>> aug =((AugmentationHolder<TrafficActionExtendedCommunity>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._sample = base.isSample();
        this._terminalAction = base.isTerminalAction();
    }


    public Boolean isSample() {
        return _sample;
    }
    
    public Boolean isTerminalAction() {
        return _terminalAction;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<TrafficActionExtendedCommunity>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    
    public TrafficActionExtendedCommunityBuilder setSample(final Boolean value) {
        this._sample = value;
        return this;
    }
    
    public TrafficActionExtendedCommunityBuilder setTerminalAction(final Boolean value) {
        this._terminalAction = value;
        return this;
    }
    
    public TrafficActionExtendedCommunityBuilder addAugmentation(Class<? extends Augmentation<TrafficActionExtendedCommunity>> augmentationType, Augmentation<TrafficActionExtendedCommunity> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public TrafficActionExtendedCommunityBuilder removeAugmentation(Class<? extends Augmentation<TrafficActionExtendedCommunity>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public TrafficActionExtendedCommunity build() {
        return new TrafficActionExtendedCommunityImpl(this);
    }

    private static final class TrafficActionExtendedCommunityImpl
        extends AbstractAugmentable<TrafficActionExtendedCommunity>
        implements TrafficActionExtendedCommunity {
    
        private final Boolean _sample;
        private final Boolean _terminalAction;
    
        TrafficActionExtendedCommunityImpl(TrafficActionExtendedCommunityBuilder base) {
            super(base.augmentation);
            this._sample = base.isSample();
            this._terminalAction = base.isTerminalAction();
        }
    
        @Override
        public Boolean isSample() {
            return _sample;
        }
        
        @Override
        public Boolean isTerminalAction() {
            return _terminalAction;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_sample);
            result = prime * result + Objects.hashCode(_terminalAction);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!TrafficActionExtendedCommunity.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            TrafficActionExtendedCommunity other = (TrafficActionExtendedCommunity)obj;
            if (!Objects.equals(_sample, other.isSample())) {
                return false;
            }
            if (!Objects.equals(_terminalAction, other.isTerminalAction())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                TrafficActionExtendedCommunityImpl otherImpl = (TrafficActionExtendedCommunityImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<TrafficActionExtendedCommunity>>, Augmentation<TrafficActionExtendedCommunity>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("TrafficActionExtendedCommunity");
            CodeHelpers.appendValue(helper, "_sample", _sample);
            CodeHelpers.appendValue(helper, "_terminalAction", _terminalAction);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
