/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.protocol.bgp.parser.impl.message;

import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.Unpooled;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import org.opendaylight.protocol.bgp.parser.BGPDocumentedException;
import org.opendaylight.protocol.bgp.parser.BGPError;
import org.opendaylight.protocol.bgp.parser.BGPParsingException;
import org.opendaylight.protocol.bgp.parser.spi.MessageParser;
import org.opendaylight.protocol.bgp.parser.spi.MessageSerializer;
import org.opendaylight.protocol.bgp.parser.spi.MessageUtil;
import org.opendaylight.protocol.bgp.parser.spi.ParameterLengthOverflowException;
import org.opendaylight.protocol.bgp.parser.spi.ParameterParser;
import org.opendaylight.protocol.bgp.parser.spi.ParameterRegistry;
import org.opendaylight.protocol.bgp.parser.spi.ParameterSerializer;
import org.opendaylight.protocol.bgp.parser.spi.PeerSpecificParserConstraint;
import org.opendaylight.protocol.util.Ipv4Util;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.inet.types.rev130715.AsNumber;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.inet.types.rev130715.Ipv4Address;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.message.rev180329.Open;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.message.rev180329.OpenBuilder;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.message.rev180329.open.message.BgpParameters;
import org.opendaylight.yangtools.yang.binding.Notification;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class BGPOpenMessageParser
implements MessageParser,
MessageSerializer {
    private static final Logger LOG = LoggerFactory.getLogger(BGPOpenMessageParser.class);
    public static final int TYPE = 1;
    private static final int VERSION_SIZE = 1;
    private static final int AS_SIZE = 2;
    private static final int HOLD_TIME_SIZE = 2;
    private static final int BGP_ID_SIZE = 4;
    private static final int OPT_PARAM_LENGTH_SIZE = 1;
    private static final int OPT_PARAM_EXT_PARAM = 255;
    private static final int MIN_MSG_LENGTH = 10;
    private static final int BGP_VERSION = 4;
    public static final int AS_TRANS = 23456;
    private final ParameterRegistry reg;

    public BGPOpenMessageParser(ParameterRegistry reg) {
        this.reg = Objects.requireNonNull(reg);
    }

    public void serializeMessage(Notification msg, ByteBuf bytes) {
        Preconditions.checkArgument((boolean)(msg instanceof Open), (String)"Message needs to be of type Open, not %s", (Object)msg);
        Open open = (Open)msg;
        ByteBuf msgBody = Unpooled.buffer().writeByte(4);
        int openAS = open.getMyAsNumber();
        if (openAS > 65535) {
            openAS = 23456;
        }
        msgBody.writeShort(openAS).writeShort(open.getHoldTimer().intValue()).writeBytes(Ipv4Util.bytesForAddress((Ipv4Address)open.getBgpIdentifier()));
        this.serializeParameters(open.getBgpParameters(), msgBody);
        MessageUtil.formatMessage((int)1, (ByteBuf)msgBody, (ByteBuf)bytes);
    }

    private void serializeParameters(List<BgpParameters> params, ByteBuf msgBody) {
        if (params == null || params.isEmpty()) {
            msgBody.writeByte(0);
            return;
        }
        ByteBuf normal = this.normalSerializeParameters(params);
        if (normal != null) {
            int length = normal.writerIndex();
            Verify.verify((length <= 255 ? 1 : 0) != 0);
            msgBody.writeByte(length);
            msgBody.writeBytes(normal);
            return;
        }
        ByteBuf buffer = Unpooled.buffer();
        for (BgpParameters param : params) {
            Optional optSer = this.reg.findSerializer(param);
            if (optSer.isPresent()) {
                ((ParameterSerializer)optSer.get()).serializeExtendedParameter(param, buffer);
                continue;
            }
            LOG.debug("Ignoring unregistered parameter {}", (Object)param);
        }
        int length = buffer.writerIndex();
        Preconditions.checkState((length <= 65535 ? 1 : 0) != 0);
        msgBody.writeByte(255);
        msgBody.writeByte(255);
        msgBody.writeShort(length);
        msgBody.writeBytes(buffer);
    }

    private ByteBuf normalSerializeParameters(List<BgpParameters> params) {
        ByteBuf buffer = Unpooled.buffer();
        for (BgpParameters param : params) {
            Optional optSer = this.reg.findSerializer(param);
            if (optSer.isPresent()) {
                try {
                    ((ParameterSerializer)optSer.get()).serializeParameter(param, buffer);
                    continue;
                }
                catch (ParameterLengthOverflowException e) {
                    LOG.debug("Forcing extended parameter serialization", (Throwable)e);
                    return null;
                }
            }
            LOG.debug("Ingnoring unregistered parameter {}", (Object)param);
        }
        int length = buffer.writerIndex();
        if (length > 255) {
            LOG.debug("Final parameter size is {}, forcing extended serialization", (Object)length);
            return null;
        }
        return buffer;
    }

    public Open parseMessageBody(ByteBuf body, int messageLength, PeerSpecificParserConstraint constraint) throws BGPDocumentedException {
        Ipv4Address bgpId;
        Preconditions.checkArgument((body != null ? 1 : 0) != 0, (Object)"Buffer cannot be null.");
        if (body.readableBytes() < 10) {
            throw BGPDocumentedException.badMessageLength((String)"Open message too small.", (int)messageLength);
        }
        short version = body.readUnsignedByte();
        if (version != 4) {
            throw new BGPDocumentedException("BGP Protocol version " + version + " not supported.", BGPError.VERSION_NOT_SUPPORTED);
        }
        AsNumber as = new AsNumber(Long.valueOf(body.readUnsignedShort()));
        int holdTime = body.readUnsignedShort();
        if (holdTime == 1 || holdTime == 2) {
            throw new BGPDocumentedException("Hold time value not acceptable.", BGPError.HOLD_TIME_NOT_ACC);
        }
        try {
            bgpId = Ipv4Util.addressForByteBuf((ByteBuf)body);
        }
        catch (IllegalArgumentException e) {
            throw new BGPDocumentedException("BGP Identifier is not a valid IPv4 Address", BGPError.BAD_BGP_ID, (Exception)e);
        }
        short optLength = body.readUnsignedByte();
        List<BgpParameters> optParams = this.parseParameters(body.slice(), optLength);
        LOG.debug("BGP Open message was parsed: AS = {}, holdTimer = {}, bgpId = {}, optParams = {}", new Object[]{as, holdTime, bgpId, optParams});
        return new OpenBuilder().setMyAsNumber(Integer.valueOf(as.getValue().intValue())).setHoldTimer(Integer.valueOf(holdTime)).setBgpIdentifier(bgpId).setBgpParameters(optParams).build();
    }

    private List<BgpParameters> parseParameters(ByteBuf buffer, int length) throws BGPDocumentedException {
        int realLength;
        OptionalInt extendedLength;
        if (length == 0) {
            return ImmutableList.of();
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace("Started parsing of BGP parameter: {} length {}", (Object)ByteBufUtil.hexDump((ByteBuf)buffer), (Object)length);
        }
        if ((extendedLength = BGPOpenMessageParser.extractExtendedLength(buffer, length)).isPresent()) {
            realLength = extendedLength.getAsInt();
            if (realLength < 255) {
                LOG.debug("Peer used Extended Optional Parameters Length to encode length {}", (Object)realLength);
            }
        } else {
            realLength = length;
        }
        if (buffer.readableBytes() > realLength) {
            buffer.writerIndex(buffer.readerIndex() + realLength);
            LOG.trace("Truncated BGP parameter buffer to length {}: {}", (Object)realLength, (Object)ByteBufUtil.hexDump((ByteBuf)buffer));
        }
        int lengthSize = extendedLength.isPresent() ? 1 : 2;
        ArrayList<BgpParameters> params = new ArrayList<BgpParameters>();
        while (buffer.isReadable()) {
            BgpParameters param;
            short paramType = buffer.readUnsignedByte();
            Optional parser = this.reg.findParser((int)paramType);
            if (!parser.isPresent()) {
                throw new BGPDocumentedException("Parameter " + paramType + " not supported", BGPError.OPT_PARAM_NOT_SUPPORTED);
            }
            if (buffer.readableBytes() <= lengthSize) {
                throw new BGPDocumentedException("Malformed parameter encountered (" + buffer.readableBytes() + " bytes left)", BGPError.UNSPECIFIC_OPEN_ERROR);
            }
            int paramLength = extendedLength.isPresent() ? buffer.readUnsignedShort() : (int)buffer.readUnsignedByte();
            ByteBuf paramBody = buffer.readSlice(paramLength);
            try {
                param = ((ParameterParser)parser.get()).parseParameter(paramBody);
            }
            catch (BGPParsingException e) {
                throw new BGPDocumentedException("Optional parameter not parsed", BGPError.UNSPECIFIC_OPEN_ERROR, (Exception)((Object)e));
            }
            params.add((BgpParameters)Verify.verifyNotNull((Object)param));
        }
        LOG.trace("Parsed BGP parameters: {}", params);
        return params;
    }

    private static OptionalInt extractExtendedLength(ByteBuf buffer, int length) throws BGPDocumentedException {
        short type = buffer.markReaderIndex().readUnsignedByte();
        if (type != 255) {
            buffer.resetReaderIndex();
            return OptionalInt.empty();
        }
        if (length != 255) {
            LOG.debug("Peer uses Extended Optional Parameters Length, but indicated RFC4271 length as {}", (Object)length);
        }
        if (length < 3) {
            throw new BGPDocumentedException("Malformed Extended Length parameter encountered (" + (length - 1) + " bytes left)", BGPError.UNSPECIFIC_OPEN_ERROR);
        }
        int avail = buffer.readableBytes();
        if (avail < 2) {
            throw new BGPDocumentedException("Buffer underrun: require 2 bytes, only " + avail + " bytes left", BGPError.UNSPECIFIC_OPEN_ERROR);
        }
        return OptionalInt.of(buffer.readUnsignedShort());
    }
}

