/*
 * Copyright (c) 2013 Cisco Systems, Inc. and others.  All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 */
package org.opendaylight.protocol.bgp.parser.impl;

import static org.junit.Assert.assertEquals;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import java.util.ArrayList;
import java.util.List;
import org.junit.Test;
import org.opendaylight.protocol.bgp.parser.BGPDocumentedException;
import org.opendaylight.protocol.bgp.parser.impl.message.BGPOpenMessageParser;
import org.opendaylight.protocol.bgp.parser.spi.pojo.ServiceLoaderBGPExtensionProviderContext;
import org.opendaylight.protocol.util.ByteArray;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.inet.types.rev130715.AsNumber;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.inet.types.rev130715.Ipv4AddressNoZone;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.message.rev200120.Open;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.message.rev200120.OpenBuilder;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.message.rev200120.ProtocolVersion;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.message.rev200120.open.message.BgpParametersBuilder;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.message.rev200120.open.message.bgp.parameters.OptionalCapabilities;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.message.rev200120.open.message.bgp.parameters.OptionalCapabilitiesBuilder;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.message.rev200120.open.message.bgp.parameters.optional.capabilities.CParametersBuilder;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.message.rev200120.open.message.bgp.parameters.optional.capabilities.c.parameters.As4BytesCapabilityBuilder;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.multiprotocol.rev180329.CParameters1;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.multiprotocol.rev180329.CParameters1Builder;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.multiprotocol.rev180329.mp.capabilities.GracefulRestartCapability;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.bgp.multiprotocol.rev180329.mp.capabilities.GracefulRestartCapabilityBuilder;
import org.opendaylight.yangtools.yang.common.Uint16;
import org.opendaylight.yangtools.yang.common.Uint32;
import org.opendaylight.yangtools.yang.common.Uint8;

public class OpenTest {

    @Test
    public void testSerializeOpen() throws BGPDocumentedException {
        final Open open = new OpenBuilder()
                .setBgpIdentifier(new Ipv4AddressNoZone("127.0.0.1"))
                .setMyAsNumber(Uint16.valueOf(30))
                .setHoldTimer(Uint16.valueOf(3))
                .setVersion(new ProtocolVersion(Uint8.valueOf(4)))
                .setBgpParameters(ImmutableList.of(new BgpParametersBuilder()
                    .setOptionalCapabilities(ImmutableList.of(new OptionalCapabilitiesBuilder()
                        .setCParameters(new CParametersBuilder()
                            .setAs4BytesCapability(new As4BytesCapabilityBuilder()
                                .setAsNumber(new AsNumber(Uint32.valueOf(1000))).build())
                            .addAugmentation(new CParameters1Builder()
                                .setGracefulRestartCapability(new GracefulRestartCapabilityBuilder()
                                    .setRestartFlags(new GracefulRestartCapability.RestartFlags(false))
                                    .setRestartTime(Uint16.ZERO)
                                    .setTables(ImmutableMap.of()).build())
                                .build())
                            .build())
                        .build())).build()))
                .build();
        final ByteBuf msg = Unpooled.buffer();
        new BGPOpenMessageParser(ServiceLoaderBGPExtensionProviderContext.getSingletonInstance().getParameterRegistry())
            .serializeMessage(open, msg);
        final byte[] temp = ByteArray.cutBytes(ByteArray.getAllBytes(msg), 19);
        final Open openResult = new BGPOpenMessageParser(ServiceLoaderBGPExtensionProviderContext.getSingletonInstance()
            .getParameterRegistry()).parseMessageBody(Unpooled.copiedBuffer(temp), temp.length, null);

        assertEquals(open.getBgpIdentifier(),openResult.getBgpIdentifier());
        assertEquals(open.getHoldTimer(), openResult.getHoldTimer());
        assertEquals(open.getMyAsNumber(), openResult.getMyAsNumber());
        boolean grace = false;
        if (openResult.getBgpParameters().get(0).getOptionalCapabilities().get(0).getCParameters()
                .getAs4BytesCapability() != null) {
            grace = true;
        }
        assertEquals(open.getBgpParameters().get(0).getOptionalCapabilities().get(0).getCParameters()
            .getAs4BytesCapability(),
            openResult.getBgpParameters().get(0).getOptionalCapabilities().get(grace ? 0 : 1).getCParameters()
            .getAs4BytesCapability());
        assertEquals(open.getBgpParameters().get(0).getOptionalCapabilities().get(0).getCParameters()
            .augmentation(CParameters1.class).getGracefulRestartCapability(),
            openResult.getBgpParameters().get(0).getOptionalCapabilities().get(grace ? 1 : 0).getCParameters()
                .augmentation(CParameters1.class).getGracefulRestartCapability());

    }

    @Test
    public void testSerializeLongOpen() throws BGPDocumentedException {
        final List<OptionalCapabilities> capabilities = new ArrayList<>();
        for (int i = 0; i < 200; ++i) {
            capabilities.add(new OptionalCapabilitiesBuilder()
                .setCParameters(new CParametersBuilder()
                    .setAs4BytesCapability(new As4BytesCapabilityBuilder()
                        .setAsNumber(new AsNumber(Uint32.valueOf(1000))).build())
                    .build())
                .build());
        }

        final Open open = new OpenBuilder()
                .setBgpIdentifier(new Ipv4AddressNoZone("127.0.0.1"))
                .setMyAsNumber(Uint16.valueOf(30))
                .setHoldTimer(Uint16.valueOf(3))
                .setVersion(new ProtocolVersion(Uint8.valueOf(4)))
                .setBgpParameters(ImmutableList.of(
                    new BgpParametersBuilder().setOptionalCapabilities(capabilities).build()))
                .build();

        final ByteBuf msg = Unpooled.buffer();
        new BGPOpenMessageParser(ServiceLoaderBGPExtensionProviderContext.getSingletonInstance().getParameterRegistry())
            .serializeMessage(open, msg);

        assertEquals(1235, msg.readableBytes());

        final byte[] temp = ByteArray.cutBytes(ByteArray.getAllBytes(msg), 19);
        final Open openResult = new BGPOpenMessageParser(ServiceLoaderBGPExtensionProviderContext.getSingletonInstance()
            .getParameterRegistry()).parseMessageBody(Unpooled.copiedBuffer(temp), temp.length, null);

        assertEquals(200, openResult.getBgpParameters().get(0).getOptionalCapabilities().size());
    }
}
