package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMap.Builder;
import java.lang.Class;
import java.lang.Integer;
import java.lang.Override;
import java.lang.String;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.eclipse.jdt.annotation.NonNull;
import org.eclipse.jdt.annotation.Nullable;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.inet.types.rev130715.IpAddress;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.vertex.Srgb;
import org.opendaylight.yangtools.yang.binding.DataObject;
import org.opendaylight.yangtools.yang.binding.Enumeration;
import org.opendaylight.yangtools.yang.common.QName;
import org.opendaylight.yangtools.yang.common.Uint32;
import org.opendaylight.yangtools.yang.common.Uint64;

/**
 * Vertex (node) representation for the network topology
 *
 * <p>
 * This class represents the following YANG schema fragment defined in module <b>graph</b>
 * <pre>
 * grouping vertex {
 *   leaf vertex-id {
 *     type uint64;
 *   }
 *   leaf name {
 *     type string;
 *   }
 *   leaf router-id {
 *     type inet:ip-address;
 *   }
 *   leaf vertex-type {
 *     type enumeration {
 *       enum standard {
 *         value 0;
 *       }
 *       enum abr {
 *         value 1;
 *       }
 *       enum asbr-in {
 *         value 2;
 *       }
 *       enum asbr-out {
 *         value 3;
 *       }
 *       enum pseudo {
 *         value 4;
 *       }
 *     }
 *     default standard;
 *   }
 *   container srgb {
 *     uses srgb;
 *   }
 *   leaf asn {
 *     type uint32;
 *   }
 * }
 * </pre>The schema path to identify an instance is
 * <i>graph/vertex</i>
 *
 */
public interface Vertex
    extends
    DataObject
{


    public enum VertexType implements Enumeration {
        Standard(0, "standard"),
        
        Abr(1, "abr"),
        
        AsbrIn(2, "asbr-in"),
        
        AsbrOut(3, "asbr-out"),
        
        Pseudo(4, "pseudo")
        ;
    
        private static final Map<String, VertexType> NAME_MAP;
        private static final Map<Integer, VertexType> VALUE_MAP;
    
        static {
            final Builder<String, VertexType> nb = ImmutableMap.builder();
            final Builder<Integer, VertexType> vb = ImmutableMap.builder();
            for (VertexType enumItem : VertexType.values()) {
                vb.put(enumItem.value, enumItem);
                nb.put(enumItem.name, enumItem);
            }
    
            NAME_MAP = nb.build();
            VALUE_MAP = vb.build();
        }
    
        private final String name;
        private final int value;
    
        private VertexType(int value, String name) {
            this.value = value;
            this.name = name;
        }
    
        @Override
        public String getName() {
            return name;
        }
    
        @Override
        public int getIntValue() {
            return value;
        }
    
        /**
         * Return the enumeration member whose {@link #getName()} matches specified value.
         *
         * @param name YANG assigned name
         * @return corresponding VertexType item, if present
         * @throws NullPointerException if name is null
         */
        public static Optional<VertexType> forName(String name) {
            return Optional.ofNullable(NAME_MAP.get(Objects.requireNonNull(name)));
        }
    
        /**
         * Return the enumeration member whose {@link #getIntValue()} matches specified value.
         *
         * @param intValue integer value
         * @return corresponding VertexType item, or null if no such item exists
         */
        public static VertexType forValue(int intValue) {
            return VALUE_MAP.get(intValue);
        }
    }

    public static final @NonNull QName QNAME = $YangModuleInfoImpl.qnameOf("vertex");

    @Override
    Class<? extends Vertex> implementedInterface();
    
    /**
     * Identifier of the Vertex
     *
     *
     *
     * @return <code>org.opendaylight.yangtools.yang.common.Uint64</code> <code>vertexId</code>, or <code>null</code> if not present
     */
    @Nullable Uint64 getVertexId();
    
    /**
     * Name of the Vertex when known
     *
     *
     *
     * @return <code>java.lang.String</code> <code>name</code>, or <code>null</code> if not present
     */
    @Nullable String getName();
    
    /**
     * Global unique IP Trafic Engineering Router ID
     *
     *
     *
     * @return <code>org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.inet.types.rev130715.IpAddress</code> <code>routerId</code>, or <code>null</code> if not present
     */
    @Nullable IpAddress getRouterId();
    
    /**
     * @return <code>org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.Vertex.VertexType</code> <code>vertexType</code>, or <code>null</code> if not present
     */
    @Nullable VertexType getVertexType();
    
    /**
     * Segment Routing Global Block
     *
     *
     *
     * @return <code>org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.vertex.Srgb</code> <code>srgb</code>, or <code>null</code> if not present
     */
    @Nullable Srgb getSrgb();
    
    /**
     * AS Number
     *
     *
     *
     * @return <code>org.opendaylight.yangtools.yang.common.Uint32</code> <code>asn</code>, or <code>null</code> if not present
     */
    @Nullable Uint32 getAsn();

}

