package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.edge.attributes;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Deprecated;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.Delay;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link MinMaxDelayBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     MinMaxDelayBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new MinMaxDelayBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of MinMaxDelayBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see MinMaxDelayBuilder
 * @see Builder
 *
 */
public class MinMaxDelayBuilder implements Builder<MinMaxDelay> {

    private Delay _maxDelay;
    private Delay _minDelay;


    Map<Class<? extends Augmentation<MinMaxDelay>>, Augmentation<MinMaxDelay>> augmentation = Collections.emptyMap();

    public MinMaxDelayBuilder() {
    }

    public MinMaxDelayBuilder(MinMaxDelay base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<MinMaxDelay>>, Augmentation<MinMaxDelay>> aug =((AugmentationHolder<MinMaxDelay>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._maxDelay = base.getMaxDelay();
        this._minDelay = base.getMinDelay();
    }


    public Delay getMaxDelay() {
        return _maxDelay;
    }
    
    public Delay getMinDelay() {
        return _minDelay;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<MinMaxDelay>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(Objects.requireNonNull(augmentationType));
    }

    
    public MinMaxDelayBuilder setMaxDelay(final Delay value) {
        this._maxDelay = value;
        return this;
    }
    
    public MinMaxDelayBuilder setMinDelay(final Delay value) {
        this._minDelay = value;
        return this;
    }
    
    /**
      * Add an augmentation to this builder's product.
      *
      * @param augmentation augmentation to be added
      * @return this builder
      * @throws NullPointerException if {@code augmentation} is null
      */
    public MinMaxDelayBuilder addAugmentation(Augmentation<MinMaxDelay> augmentation) {
        return doAddAugmentation(augmentation.implementedInterface(), augmentation);
    }
    
    /**
      * Add or remove an augmentation to this builder's product.
      *
      * @param augmentationType augmentation type to be added or removed
      * @param augmentationValue augmentation value, null if the augmentation type should be removed
      * @return this builder
      * @deprecated Use either {@link #addAugmentation(Augmentation)} or {@link #removeAugmentation(Class)} instead.
      */
    @Deprecated(forRemoval = true)
    public MinMaxDelayBuilder addAugmentation(Class<? extends Augmentation<MinMaxDelay>> augmentationType, Augmentation<MinMaxDelay> augmentationValue) {
        return augmentationValue == null ? removeAugmentation(augmentationType) : doAddAugmentation(augmentationType, augmentationValue);
    }
    
    /**
      * Remove an augmentation from this builder's product. If this builder does not track such an augmentation
      * type, this method does nothing.
      *
      * @param augmentationType augmentation type to be removed
      * @return this builder
      */
    public MinMaxDelayBuilder removeAugmentation(Class<? extends Augmentation<MinMaxDelay>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }
    
    private MinMaxDelayBuilder doAddAugmentation(Class<? extends Augmentation<MinMaxDelay>> augmentationType, Augmentation<MinMaxDelay> augmentationValue) {
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }

    @Override
    public MinMaxDelay build() {
        return new MinMaxDelayImpl(this);
    }

    private static final class MinMaxDelayImpl
        extends AbstractAugmentable<MinMaxDelay>
        implements MinMaxDelay {
    
        private final Delay _maxDelay;
        private final Delay _minDelay;
    
        MinMaxDelayImpl(MinMaxDelayBuilder base) {
            super(base.augmentation);
            this._maxDelay = base.getMaxDelay();
            this._minDelay = base.getMinDelay();
        }
    
        @Override
        public Delay getMaxDelay() {
            return _maxDelay;
        }
        
        @Override
        public Delay getMinDelay() {
            return _minDelay;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_maxDelay);
            result = prime * result + Objects.hashCode(_minDelay);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!MinMaxDelay.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            MinMaxDelay other = (MinMaxDelay)obj;
            if (!Objects.equals(_maxDelay, other.getMaxDelay())) {
                return false;
            }
            if (!Objects.equals(_minDelay, other.getMinDelay())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                MinMaxDelayImpl otherImpl = (MinMaxDelayImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<MinMaxDelay>>, Augmentation<MinMaxDelay>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("MinMaxDelay");
            CodeHelpers.appendValue(helper, "_maxDelay", _maxDelay);
            CodeHelpers.appendValue(helper, "_minDelay", _minDelay);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
