package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.edge.attributes;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Deprecated;
import java.lang.Object;
import java.lang.Override;
import java.lang.Short;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.DecimalBandwidth;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;
import org.opendaylight.yangtools.yang.common.Uint8;

/**
 * Class that builds {@link UnreservedBandwidthBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     UnreservedBandwidthBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new UnreservedBandwidthBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of UnreservedBandwidthBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see UnreservedBandwidthBuilder
 * @see Builder
 *
 */
public class UnreservedBandwidthBuilder implements Builder<UnreservedBandwidth> {

    private DecimalBandwidth _bandwidth;
    private Uint8 _classType;
    private UnreservedBandwidthKey key;


    Map<Class<? extends Augmentation<UnreservedBandwidth>>, Augmentation<UnreservedBandwidth>> augmentation = Collections.emptyMap();

    public UnreservedBandwidthBuilder() {
    }

    public UnreservedBandwidthBuilder(UnreservedBandwidth base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<UnreservedBandwidth>>, Augmentation<UnreservedBandwidth>> aug =((AugmentationHolder<UnreservedBandwidth>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this.key = base.key();
        this._classType = base.getClassType();
        this._bandwidth = base.getBandwidth();
    }


    public UnreservedBandwidthKey key() {
        return key;
    }
    
    public DecimalBandwidth getBandwidth() {
        return _bandwidth;
    }
    
    public Uint8 getClassType() {
        return _classType;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<UnreservedBandwidth>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(Objects.requireNonNull(augmentationType));
    }

    public UnreservedBandwidthBuilder withKey(final UnreservedBandwidthKey key) {
        this.key = key;
        return this;
    }
    
    public UnreservedBandwidthBuilder setBandwidth(final DecimalBandwidth value) {
        this._bandwidth = value;
        return this;
    }
    
    private static void checkClassTypeRange(final short value) {
        if (value <= (short)7) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..7]]", value);
    }
    
    public UnreservedBandwidthBuilder setClassType(final Uint8 value) {
        if (value != null) {
            checkClassTypeRange(value.shortValue());
            
        }
        this._classType = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setClassType(Uint8)} instead.
     */
    @Deprecated(forRemoval = true)
    public UnreservedBandwidthBuilder setClassType(final Short value) {
        return setClassType(CodeHelpers.compatUint(value));
    }
    
    /**
      * Add an augmentation to this builder's product.
      *
      * @param augmentation augmentation to be added
      * @return this builder
      * @throws NullPointerException if {@code augmentation} is null
      */
    public UnreservedBandwidthBuilder addAugmentation(Augmentation<UnreservedBandwidth> augmentation) {
        return doAddAugmentation(augmentation.implementedInterface(), augmentation);
    }
    
    /**
      * Add or remove an augmentation to this builder's product.
      *
      * @param augmentationType augmentation type to be added or removed
      * @param augmentationValue augmentation value, null if the augmentation type should be removed
      * @return this builder
      * @deprecated Use either {@link #addAugmentation(Augmentation)} or {@link #removeAugmentation(Class)} instead.
      */
    @Deprecated(forRemoval = true)
    public UnreservedBandwidthBuilder addAugmentation(Class<? extends Augmentation<UnreservedBandwidth>> augmentationType, Augmentation<UnreservedBandwidth> augmentationValue) {
        return augmentationValue == null ? removeAugmentation(augmentationType) : doAddAugmentation(augmentationType, augmentationValue);
    }
    
    /**
      * Remove an augmentation from this builder's product. If this builder does not track such an augmentation
      * type, this method does nothing.
      *
      * @param augmentationType augmentation type to be removed
      * @return this builder
      */
    public UnreservedBandwidthBuilder removeAugmentation(Class<? extends Augmentation<UnreservedBandwidth>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }
    
    private UnreservedBandwidthBuilder doAddAugmentation(Class<? extends Augmentation<UnreservedBandwidth>> augmentationType, Augmentation<UnreservedBandwidth> augmentationValue) {
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }

    @Override
    public UnreservedBandwidth build() {
        return new UnreservedBandwidthImpl(this);
    }

    private static final class UnreservedBandwidthImpl
        extends AbstractAugmentable<UnreservedBandwidth>
        implements UnreservedBandwidth {
    
        private final DecimalBandwidth _bandwidth;
        private final Uint8 _classType;
        private final UnreservedBandwidthKey key;
    
        UnreservedBandwidthImpl(UnreservedBandwidthBuilder base) {
            super(base.augmentation);
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new UnreservedBandwidthKey(base.getClassType());
            }
            this._classType = key.getClassType();
            this._bandwidth = base.getBandwidth();
        }
    
        @Override
        public UnreservedBandwidthKey key() {
            return key;
        }
        
        @Override
        public DecimalBandwidth getBandwidth() {
            return _bandwidth;
        }
        
        @Override
        public Uint8 getClassType() {
            return _classType;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_bandwidth);
            result = prime * result + Objects.hashCode(_classType);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!UnreservedBandwidth.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            UnreservedBandwidth other = (UnreservedBandwidth)obj;
            if (!Objects.equals(_bandwidth, other.getBandwidth())) {
                return false;
            }
            if (!Objects.equals(_classType, other.getClassType())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                UnreservedBandwidthImpl otherImpl = (UnreservedBandwidthImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<UnreservedBandwidth>>, Augmentation<UnreservedBandwidth>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("UnreservedBandwidth");
            CodeHelpers.appendValue(helper, "_bandwidth", _bandwidth);
            CodeHelpers.appendValue(helper, "_classType", _classType);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
