package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.graph.topology;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Deprecated;
import java.lang.Long;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.graph.topology.graph.Edge;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.graph.topology.graph.EdgeKey;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.graph.topology.graph.Prefix;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.graph.topology.graph.PrefixKey;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.graph.topology.graph.Vertex;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.graph.topology.graph.VertexKey;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;
import org.opendaylight.yangtools.yang.common.Uint32;

/**
 * Class that builds {@link GraphBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     GraphBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new GraphBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of GraphBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see GraphBuilder
 * @see Builder
 *
 */
public class GraphBuilder implements Builder<Graph> {

    private Uint32 _asn;
    private Graph.DomainScope _domainScope;
    private Map<EdgeKey, Edge> _edge;
    private String _name;
    private Map<PrefixKey, Prefix> _prefix;
    private Map<VertexKey, Vertex> _vertex;
    private GraphKey key;


    Map<Class<? extends Augmentation<Graph>>, Augmentation<Graph>> augmentation = Collections.emptyMap();

    public GraphBuilder() {
    }

    public GraphBuilder(Graph base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<Graph>>, Augmentation<Graph>> aug =((AugmentationHolder<Graph>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this.key = base.key();
        this._name = base.getName();
        this._asn = base.getAsn();
        this._domainScope = base.getDomainScope();
        this._edge = base.getEdge();
        this._prefix = base.getPrefix();
        this._vertex = base.getVertex();
    }


    public GraphKey key() {
        return key;
    }
    
    public Uint32 getAsn() {
        return _asn;
    }
    
    public Graph.DomainScope getDomainScope() {
        return _domainScope;
    }
    
    public Map<EdgeKey, Edge> getEdge() {
        return _edge;
    }
    
    public String getName() {
        return _name;
    }
    
    public Map<PrefixKey, Prefix> getPrefix() {
        return _prefix;
    }
    
    public Map<VertexKey, Vertex> getVertex() {
        return _vertex;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<Graph>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(Objects.requireNonNull(augmentationType));
    }

    public GraphBuilder withKey(final GraphKey key) {
        this.key = key;
        return this;
    }
    
    public GraphBuilder setAsn(final Uint32 value) {
        this._asn = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setAsn(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public GraphBuilder setAsn(final Long value) {
        return setAsn(CodeHelpers.compatUint(value));
    }
    
    public GraphBuilder setDomainScope(final Graph.DomainScope value) {
        this._domainScope = value;
        return this;
    }
    public GraphBuilder setEdge(final Map<EdgeKey, Edge> values) {
        this._edge = values;
        return this;
    }
    
    /**
      * Utility migration setter.
      *
      * <b>IMPORTANT NOTE</b>: This method does not completely match previous mechanics, as the list is processed as
      *                        during this method's execution. Any future modifications of the list are <b>NOT</b>
      *                        reflected in this builder nor its products.
      *
      * @param values Legacy List of values
      * @return this builder
      * @throws IllegalArgumentException if the list contains entries with the same key
      * @throws NullPointerException if the list contains a null entry
      * @deprecated Use {#link #setEdge(Map)} instead.
      */
    @Deprecated(forRemoval = true)
    public GraphBuilder setEdge(final List<Edge> values) {
        return setEdge(CodeHelpers.compatMap(values));
    }
    
    public GraphBuilder setName(final String value) {
        this._name = value;
        return this;
    }
    public GraphBuilder setPrefix(final Map<PrefixKey, Prefix> values) {
        this._prefix = values;
        return this;
    }
    
    /**
      * Utility migration setter.
      *
      * <b>IMPORTANT NOTE</b>: This method does not completely match previous mechanics, as the list is processed as
      *                        during this method's execution. Any future modifications of the list are <b>NOT</b>
      *                        reflected in this builder nor its products.
      *
      * @param values Legacy List of values
      * @return this builder
      * @throws IllegalArgumentException if the list contains entries with the same key
      * @throws NullPointerException if the list contains a null entry
      * @deprecated Use {#link #setPrefix(Map)} instead.
      */
    @Deprecated(forRemoval = true)
    public GraphBuilder setPrefix(final List<Prefix> values) {
        return setPrefix(CodeHelpers.compatMap(values));
    }
    public GraphBuilder setVertex(final Map<VertexKey, Vertex> values) {
        this._vertex = values;
        return this;
    }
    
    /**
      * Utility migration setter.
      *
      * <b>IMPORTANT NOTE</b>: This method does not completely match previous mechanics, as the list is processed as
      *                        during this method's execution. Any future modifications of the list are <b>NOT</b>
      *                        reflected in this builder nor its products.
      *
      * @param values Legacy List of values
      * @return this builder
      * @throws IllegalArgumentException if the list contains entries with the same key
      * @throws NullPointerException if the list contains a null entry
      * @deprecated Use {#link #setVertex(Map)} instead.
      */
    @Deprecated(forRemoval = true)
    public GraphBuilder setVertex(final List<Vertex> values) {
        return setVertex(CodeHelpers.compatMap(values));
    }
    
    /**
      * Add an augmentation to this builder's product.
      *
      * @param augmentation augmentation to be added
      * @return this builder
      * @throws NullPointerException if {@code augmentation} is null
      */
    public GraphBuilder addAugmentation(Augmentation<Graph> augmentation) {
        return doAddAugmentation(augmentation.implementedInterface(), augmentation);
    }
    
    /**
      * Add or remove an augmentation to this builder's product.
      *
      * @param augmentationType augmentation type to be added or removed
      * @param augmentationValue augmentation value, null if the augmentation type should be removed
      * @return this builder
      * @deprecated Use either {@link #addAugmentation(Augmentation)} or {@link #removeAugmentation(Class)} instead.
      */
    @Deprecated(forRemoval = true)
    public GraphBuilder addAugmentation(Class<? extends Augmentation<Graph>> augmentationType, Augmentation<Graph> augmentationValue) {
        return augmentationValue == null ? removeAugmentation(augmentationType) : doAddAugmentation(augmentationType, augmentationValue);
    }
    
    /**
      * Remove an augmentation from this builder's product. If this builder does not track such an augmentation
      * type, this method does nothing.
      *
      * @param augmentationType augmentation type to be removed
      * @return this builder
      */
    public GraphBuilder removeAugmentation(Class<? extends Augmentation<Graph>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }
    
    private GraphBuilder doAddAugmentation(Class<? extends Augmentation<Graph>> augmentationType, Augmentation<Graph> augmentationValue) {
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }

    @Override
    public Graph build() {
        return new GraphImpl(this);
    }

    private static final class GraphImpl
        extends AbstractAugmentable<Graph>
        implements Graph {
    
        private final Uint32 _asn;
        private final Graph.DomainScope _domainScope;
        private final Map<EdgeKey, Edge> _edge;
        private final String _name;
        private final Map<PrefixKey, Prefix> _prefix;
        private final Map<VertexKey, Vertex> _vertex;
        private final GraphKey key;
    
        GraphImpl(GraphBuilder base) {
            super(base.augmentation);
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new GraphKey(base.getName());
            }
            this._name = key.getName();
            this._asn = base.getAsn();
            this._domainScope = base.getDomainScope();
            this._edge = CodeHelpers.emptyToNull(base.getEdge());
            this._prefix = CodeHelpers.emptyToNull(base.getPrefix());
            this._vertex = CodeHelpers.emptyToNull(base.getVertex());
        }
    
        @Override
        public GraphKey key() {
            return key;
        }
        
        @Override
        public Uint32 getAsn() {
            return _asn;
        }
        
        @Override
        public Graph.DomainScope getDomainScope() {
            return _domainScope;
        }
        
        @Override
        public Map<EdgeKey, Edge> getEdge() {
            return _edge;
        }
        
        @Override
        public String getName() {
            return _name;
        }
        
        @Override
        public Map<PrefixKey, Prefix> getPrefix() {
            return _prefix;
        }
        
        @Override
        public Map<VertexKey, Vertex> getVertex() {
            return _vertex;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_asn);
            result = prime * result + Objects.hashCode(_domainScope);
            result = prime * result + Objects.hashCode(_edge);
            result = prime * result + Objects.hashCode(_name);
            result = prime * result + Objects.hashCode(_prefix);
            result = prime * result + Objects.hashCode(_vertex);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!Graph.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            Graph other = (Graph)obj;
            if (!Objects.equals(_asn, other.getAsn())) {
                return false;
            }
            if (!Objects.equals(_domainScope, other.getDomainScope())) {
                return false;
            }
            if (!Objects.equals(_edge, other.getEdge())) {
                return false;
            }
            if (!Objects.equals(_name, other.getName())) {
                return false;
            }
            if (!Objects.equals(_prefix, other.getPrefix())) {
                return false;
            }
            if (!Objects.equals(_vertex, other.getVertex())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                GraphImpl otherImpl = (GraphImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<Graph>>, Augmentation<Graph>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("Graph");
            CodeHelpers.appendValue(helper, "_asn", _asn);
            CodeHelpers.appendValue(helper, "_domainScope", _domainScope);
            CodeHelpers.appendValue(helper, "_edge", _edge);
            CodeHelpers.appendValue(helper, "_name", _name);
            CodeHelpers.appendValue(helper, "_prefix", _prefix);
            CodeHelpers.appendValue(helper, "_vertex", _vertex);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
