package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.graph.topology.graph;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Deprecated;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.math.BigInteger;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.edge.EdgeAttributes;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;
import org.opendaylight.yangtools.yang.common.Uint64;

/**
 * Class that builds {@link EdgeBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     EdgeBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new EdgeBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of EdgeBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see EdgeBuilder
 * @see Builder
 *
 */
public class EdgeBuilder implements Builder<Edge> {

    private EdgeAttributes _edgeAttributes;
    private Uint64 _edgeId;
    private Uint64 _localVertexId;
    private String _name;
    private Uint64 _remoteVertexId;
    private EdgeKey key;


    Map<Class<? extends Augmentation<Edge>>, Augmentation<Edge>> augmentation = Collections.emptyMap();

    public EdgeBuilder() {
    }
    public EdgeBuilder(org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.Edge arg) {
        this._edgeId = arg.getEdgeId();
        this._localVertexId = arg.getLocalVertexId();
        this._remoteVertexId = arg.getRemoteVertexId();
        this._name = arg.getName();
        this._edgeAttributes = arg.getEdgeAttributes();
    }

    public EdgeBuilder(Edge base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<Edge>>, Augmentation<Edge>> aug =((AugmentationHolder<Edge>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this.key = base.key();
        this._edgeId = base.getEdgeId();
        this._edgeAttributes = base.getEdgeAttributes();
        this._localVertexId = base.getLocalVertexId();
        this._name = base.getName();
        this._remoteVertexId = base.getRemoteVertexId();
    }

    /**
     * Set fields from given grouping argument. Valid argument is instance of one of following types:
     * <ul>
     * <li>org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.Edge</li>
     * </ul>
     *
     * @param arg grouping object
     * @throws IllegalArgumentException if given argument is none of valid types
    */
    public void fieldsFrom(DataObject arg) {
        boolean isValidArg = false;
        if (arg instanceof org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.Edge) {
            this._edgeId = ((org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.Edge)arg).getEdgeId();
            this._localVertexId = ((org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.Edge)arg).getLocalVertexId();
            this._remoteVertexId = ((org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.Edge)arg).getRemoteVertexId();
            this._name = ((org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.Edge)arg).getName();
            this._edgeAttributes = ((org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.Edge)arg).getEdgeAttributes();
            isValidArg = true;
        }
        CodeHelpers.validValue(isValidArg, arg, "[org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.graph.rev191125.Edge]");
    }

    public EdgeKey key() {
        return key;
    }
    
    public EdgeAttributes getEdgeAttributes() {
        return _edgeAttributes;
    }
    
    public Uint64 getEdgeId() {
        return _edgeId;
    }
    
    public Uint64 getLocalVertexId() {
        return _localVertexId;
    }
    
    public String getName() {
        return _name;
    }
    
    public Uint64 getRemoteVertexId() {
        return _remoteVertexId;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<Edge>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(Objects.requireNonNull(augmentationType));
    }

    public EdgeBuilder withKey(final EdgeKey key) {
        this.key = key;
        return this;
    }
    
    public EdgeBuilder setEdgeAttributes(final EdgeAttributes value) {
        this._edgeAttributes = value;
        return this;
    }
    
    public EdgeBuilder setEdgeId(final Uint64 value) {
        this._edgeId = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setEdgeId(Uint64)} instead.
     */
    @Deprecated(forRemoval = true)
    public EdgeBuilder setEdgeId(final BigInteger value) {
        return setEdgeId(CodeHelpers.compatUint(value));
    }
    
    public EdgeBuilder setLocalVertexId(final Uint64 value) {
        this._localVertexId = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setLocalVertexId(Uint64)} instead.
     */
    @Deprecated(forRemoval = true)
    public EdgeBuilder setLocalVertexId(final BigInteger value) {
        return setLocalVertexId(CodeHelpers.compatUint(value));
    }
    
    public EdgeBuilder setName(final String value) {
        this._name = value;
        return this;
    }
    
    public EdgeBuilder setRemoteVertexId(final Uint64 value) {
        this._remoteVertexId = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setRemoteVertexId(Uint64)} instead.
     */
    @Deprecated(forRemoval = true)
    public EdgeBuilder setRemoteVertexId(final BigInteger value) {
        return setRemoteVertexId(CodeHelpers.compatUint(value));
    }
    
    /**
      * Add an augmentation to this builder's product.
      *
      * @param augmentation augmentation to be added
      * @return this builder
      * @throws NullPointerException if {@code augmentation} is null
      */
    public EdgeBuilder addAugmentation(Augmentation<Edge> augmentation) {
        return doAddAugmentation(augmentation.implementedInterface(), augmentation);
    }
    
    /**
      * Add or remove an augmentation to this builder's product.
      *
      * @param augmentationType augmentation type to be added or removed
      * @param augmentationValue augmentation value, null if the augmentation type should be removed
      * @return this builder
      * @deprecated Use either {@link #addAugmentation(Augmentation)} or {@link #removeAugmentation(Class)} instead.
      */
    @Deprecated(forRemoval = true)
    public EdgeBuilder addAugmentation(Class<? extends Augmentation<Edge>> augmentationType, Augmentation<Edge> augmentationValue) {
        return augmentationValue == null ? removeAugmentation(augmentationType) : doAddAugmentation(augmentationType, augmentationValue);
    }
    
    /**
      * Remove an augmentation from this builder's product. If this builder does not track such an augmentation
      * type, this method does nothing.
      *
      * @param augmentationType augmentation type to be removed
      * @return this builder
      */
    public EdgeBuilder removeAugmentation(Class<? extends Augmentation<Edge>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }
    
    private EdgeBuilder doAddAugmentation(Class<? extends Augmentation<Edge>> augmentationType, Augmentation<Edge> augmentationValue) {
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }

    @Override
    public Edge build() {
        return new EdgeImpl(this);
    }

    private static final class EdgeImpl
        extends AbstractAugmentable<Edge>
        implements Edge {
    
        private final EdgeAttributes _edgeAttributes;
        private final Uint64 _edgeId;
        private final Uint64 _localVertexId;
        private final String _name;
        private final Uint64 _remoteVertexId;
        private final EdgeKey key;
    
        EdgeImpl(EdgeBuilder base) {
            super(base.augmentation);
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new EdgeKey(base.getEdgeId());
            }
            this._edgeId = key.getEdgeId();
            this._edgeAttributes = base.getEdgeAttributes();
            this._localVertexId = base.getLocalVertexId();
            this._name = base.getName();
            this._remoteVertexId = base.getRemoteVertexId();
        }
    
        @Override
        public EdgeKey key() {
            return key;
        }
        
        @Override
        public EdgeAttributes getEdgeAttributes() {
            return _edgeAttributes;
        }
        
        @Override
        public Uint64 getEdgeId() {
            return _edgeId;
        }
        
        @Override
        public Uint64 getLocalVertexId() {
            return _localVertexId;
        }
        
        @Override
        public String getName() {
            return _name;
        }
        
        @Override
        public Uint64 getRemoteVertexId() {
            return _remoteVertexId;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_edgeAttributes);
            result = prime * result + Objects.hashCode(_edgeId);
            result = prime * result + Objects.hashCode(_localVertexId);
            result = prime * result + Objects.hashCode(_name);
            result = prime * result + Objects.hashCode(_remoteVertexId);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!Edge.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            Edge other = (Edge)obj;
            if (!Objects.equals(_edgeAttributes, other.getEdgeAttributes())) {
                return false;
            }
            if (!Objects.equals(_edgeId, other.getEdgeId())) {
                return false;
            }
            if (!Objects.equals(_localVertexId, other.getLocalVertexId())) {
                return false;
            }
            if (!Objects.equals(_name, other.getName())) {
                return false;
            }
            if (!Objects.equals(_remoteVertexId, other.getRemoteVertexId())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                EdgeImpl otherImpl = (EdgeImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<Edge>>, Augmentation<Edge>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("Edge");
            CodeHelpers.appendValue(helper, "_edgeAttributes", _edgeAttributes);
            CodeHelpers.appendValue(helper, "_edgeId", _edgeId);
            CodeHelpers.appendValue(helper, "_localVertexId", _localVertexId);
            CodeHelpers.appendValue(helper, "_name", _name);
            CodeHelpers.appendValue(helper, "_remoteVertexId", _remoteVertexId);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
