package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.controller.pcep.app.config.rev160707;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableMap;
import java.lang.Class;
import java.lang.Integer;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link PcepSessionConfigBuilder} instances.
 *
 * @see PcepSessionConfigBuilder
 *
 */
public class PcepSessionConfigBuilder implements Builder<PcepSessionConfig> {

    private Integer _deadTimerValue;
    private Integer _keepAliveTimerValue;


    Map<Class<? extends Augmentation<PcepSessionConfig>>, Augmentation<PcepSessionConfig>> augmentation = Collections.emptyMap();

    public PcepSessionConfigBuilder() {
    }

    public PcepSessionConfigBuilder(PcepSessionConfig base) {
        this._deadTimerValue = base.getDeadTimerValue();
        this._keepAliveTimerValue = base.getKeepAliveTimerValue();
        if (base instanceof PcepSessionConfigImpl) {
            PcepSessionConfigImpl impl = (PcepSessionConfigImpl) base;
            if (!impl.augmentation.isEmpty()) {
                this.augmentation = new HashMap<>(impl.augmentation);
            }
        } else if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            AugmentationHolder<PcepSessionConfig> casted =(AugmentationHolder<PcepSessionConfig>) base;
            if (!casted.augmentations().isEmpty()) {
                this.augmentation = new HashMap<>(casted.augmentations());
            }
        }
    }


    public Integer getDeadTimerValue() {
        return _deadTimerValue;
    }
    
    public Integer getKeepAliveTimerValue() {
        return _keepAliveTimerValue;
    }
    
    @SuppressWarnings("unchecked")
    public <E extends Augmentation<PcepSessionConfig>> E augmentation(Class<E> augmentationType) {
        return (E) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    private static void checkDeadTimerValueRange(final int value) {
        if (value >= 0 && value <= 65535) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..65535]]", value);
    }
    
    public PcepSessionConfigBuilder setDeadTimerValue(final Integer value) {
    if (value != null) {
        checkDeadTimerValueRange(value);
        
    }
        this._deadTimerValue = value;
        return this;
    }
    private static void checkKeepAliveTimerValueRange(final int value) {
        if (value >= 0 && value <= 65535) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..65535]]", value);
    }
    
    public PcepSessionConfigBuilder setKeepAliveTimerValue(final Integer value) {
    if (value != null) {
        checkKeepAliveTimerValueRange(value);
        
    }
        this._keepAliveTimerValue = value;
        return this;
    }
    
    public PcepSessionConfigBuilder addAugmentation(Class<? extends Augmentation<PcepSessionConfig>> augmentationType, Augmentation<PcepSessionConfig> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public PcepSessionConfigBuilder removeAugmentation(Class<? extends Augmentation<PcepSessionConfig>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public PcepSessionConfig build() {
        return new PcepSessionConfigImpl(this);
    }

    private static final class PcepSessionConfigImpl implements PcepSessionConfig {
    
        private final Integer _deadTimerValue;
        private final Integer _keepAliveTimerValue;
    
        private Map<Class<? extends Augmentation<PcepSessionConfig>>, Augmentation<PcepSessionConfig>> augmentation = Collections.emptyMap();
    
        private PcepSessionConfigImpl(PcepSessionConfigBuilder base) {
            this._deadTimerValue = base.getDeadTimerValue();
            this._keepAliveTimerValue = base.getKeepAliveTimerValue();
            this.augmentation = ImmutableMap.copyOf(base.augmentation);
        }
    
        @Override
        public Class<PcepSessionConfig> getImplementedInterface() {
            return PcepSessionConfig.class;
        }
    
        @Override
        public Integer getDeadTimerValue() {
            return _deadTimerValue;
        }
        
        @Override
        public Integer getKeepAliveTimerValue() {
            return _keepAliveTimerValue;
        }
        
        @SuppressWarnings("unchecked")
        @Override
        public <E extends Augmentation<PcepSessionConfig>> E augmentation(Class<E> augmentationType) {
            return (E) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_deadTimerValue);
            result = prime * result + Objects.hashCode(_keepAliveTimerValue);
            result = prime * result + Objects.hashCode(augmentation);
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!PcepSessionConfig.class.equals(((DataObject)obj).getImplementedInterface())) {
                return false;
            }
            PcepSessionConfig other = (PcepSessionConfig)obj;
            if (!Objects.equals(_deadTimerValue, other.getDeadTimerValue())) {
                return false;
            }
            if (!Objects.equals(_keepAliveTimerValue, other.getKeepAliveTimerValue())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                PcepSessionConfigImpl otherImpl = (PcepSessionConfigImpl) obj;
                if (!Objects.equals(augmentation, otherImpl.augmentation)) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<PcepSessionConfig>>, Augmentation<PcepSessionConfig>> e : augmentation.entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("PcepSessionConfig");
            CodeHelpers.appendValue(helper, "_deadTimerValue", _deadTimerValue);
            CodeHelpers.appendValue(helper, "_keepAliveTimerValue", _keepAliveTimerValue);
            CodeHelpers.appendValue(helper, "augmentation", augmentation.values());
            return helper.toString();
        }
    }
}
