package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.controller.pcep.app.config.rev160707;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Integer;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.controller.pcep.app.config.rev160707.pcep.dispatcher.config.Tls;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link PcepDispatcherConfigBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     PcepDispatcherConfigBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new PcepDispatcherConfigBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of PcepDispatcherConfigBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see PcepDispatcherConfigBuilder
 * @see Builder
 *
 */
public class PcepDispatcherConfigBuilder implements Builder<PcepDispatcherConfig> {

    private Integer _maxUnknownMessages;
    private Tls _tls;


    Map<Class<? extends Augmentation<PcepDispatcherConfig>>, Augmentation<PcepDispatcherConfig>> augmentation = Collections.emptyMap();

    public PcepDispatcherConfigBuilder() {
    }

    public PcepDispatcherConfigBuilder(PcepDispatcherConfig base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<PcepDispatcherConfig>>, Augmentation<PcepDispatcherConfig>> aug =((AugmentationHolder<PcepDispatcherConfig>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._maxUnknownMessages = base.getMaxUnknownMessages();
        this._tls = base.getTls();
    }


    public Integer getMaxUnknownMessages() {
        return _maxUnknownMessages;
    }
    
    public Tls getTls() {
        return _tls;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<PcepDispatcherConfig>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    private static void checkMaxUnknownMessagesRange(final int value) {
        if (value >= 1 && value <= 65535) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[1..65535]]", value);
    }
    
    public PcepDispatcherConfigBuilder setMaxUnknownMessages(final Integer value) {
    if (value != null) {
        checkMaxUnknownMessagesRange(value);
        
    }
        this._maxUnknownMessages = value;
        return this;
    }
    
    public PcepDispatcherConfigBuilder setTls(final Tls value) {
        this._tls = value;
        return this;
    }
    
    public PcepDispatcherConfigBuilder addAugmentation(Class<? extends Augmentation<PcepDispatcherConfig>> augmentationType, Augmentation<PcepDispatcherConfig> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public PcepDispatcherConfigBuilder removeAugmentation(Class<? extends Augmentation<PcepDispatcherConfig>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public PcepDispatcherConfig build() {
        return new PcepDispatcherConfigImpl(this);
    }

    private static final class PcepDispatcherConfigImpl
        extends AbstractAugmentable<PcepDispatcherConfig>
        implements PcepDispatcherConfig {
    
        private final Integer _maxUnknownMessages;
        private final Tls _tls;
    
        PcepDispatcherConfigImpl(PcepDispatcherConfigBuilder base) {
            super(base.augmentation);
            this._maxUnknownMessages = base.getMaxUnknownMessages();
            this._tls = base.getTls();
        }
    
        @Override
        public Integer getMaxUnknownMessages() {
            return _maxUnknownMessages;
        }
        
        @Override
        public Tls getTls() {
            return _tls;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_maxUnknownMessages);
            result = prime * result + Objects.hashCode(_tls);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!PcepDispatcherConfig.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            PcepDispatcherConfig other = (PcepDispatcherConfig)obj;
            if (!Objects.equals(_maxUnknownMessages, other.getMaxUnknownMessages())) {
                return false;
            }
            if (!Objects.equals(_tls, other.getTls())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                PcepDispatcherConfigImpl otherImpl = (PcepDispatcherConfigImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<PcepDispatcherConfig>>, Augmentation<PcepDispatcherConfig>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("PcepDispatcherConfig");
            CodeHelpers.appendValue(helper, "_maxUnknownMessages", _maxUnknownMessages);
            CodeHelpers.appendValue(helper, "_tls", _tls);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
