package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.controller.pcep.app.config.rev160707;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Integer;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link PcepSessionConfigBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     PcepSessionConfigBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new PcepSessionConfigBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of PcepSessionConfigBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see PcepSessionConfigBuilder
 * @see Builder
 *
 */
public class PcepSessionConfigBuilder implements Builder<PcepSessionConfig> {

    private Integer _deadTimerValue;
    private Integer _keepAliveTimerValue;


    Map<Class<? extends Augmentation<PcepSessionConfig>>, Augmentation<PcepSessionConfig>> augmentation = Collections.emptyMap();

    public PcepSessionConfigBuilder() {
    }

    public PcepSessionConfigBuilder(PcepSessionConfig base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<PcepSessionConfig>>, Augmentation<PcepSessionConfig>> aug =((AugmentationHolder<PcepSessionConfig>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._deadTimerValue = base.getDeadTimerValue();
        this._keepAliveTimerValue = base.getKeepAliveTimerValue();
    }


    public Integer getDeadTimerValue() {
        return _deadTimerValue;
    }
    
    public Integer getKeepAliveTimerValue() {
        return _keepAliveTimerValue;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<PcepSessionConfig>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    private static void checkDeadTimerValueRange(final int value) {
        if (value >= 0 && value <= 65535) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..65535]]", value);
    }
    
    public PcepSessionConfigBuilder setDeadTimerValue(final Integer value) {
    if (value != null) {
        checkDeadTimerValueRange(value);
        
    }
        this._deadTimerValue = value;
        return this;
    }
    private static void checkKeepAliveTimerValueRange(final int value) {
        if (value >= 0 && value <= 65535) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..65535]]", value);
    }
    
    public PcepSessionConfigBuilder setKeepAliveTimerValue(final Integer value) {
    if (value != null) {
        checkKeepAliveTimerValueRange(value);
        
    }
        this._keepAliveTimerValue = value;
        return this;
    }
    
    public PcepSessionConfigBuilder addAugmentation(Class<? extends Augmentation<PcepSessionConfig>> augmentationType, Augmentation<PcepSessionConfig> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public PcepSessionConfigBuilder removeAugmentation(Class<? extends Augmentation<PcepSessionConfig>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public PcepSessionConfig build() {
        return new PcepSessionConfigImpl(this);
    }

    private static final class PcepSessionConfigImpl
        extends AbstractAugmentable<PcepSessionConfig>
        implements PcepSessionConfig {
    
        private final Integer _deadTimerValue;
        private final Integer _keepAliveTimerValue;
    
        PcepSessionConfigImpl(PcepSessionConfigBuilder base) {
            super(base.augmentation);
            this._deadTimerValue = base.getDeadTimerValue();
            this._keepAliveTimerValue = base.getKeepAliveTimerValue();
        }
    
        @Override
        public Integer getDeadTimerValue() {
            return _deadTimerValue;
        }
        
        @Override
        public Integer getKeepAliveTimerValue() {
            return _keepAliveTimerValue;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_deadTimerValue);
            result = prime * result + Objects.hashCode(_keepAliveTimerValue);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!PcepSessionConfig.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            PcepSessionConfig other = (PcepSessionConfig)obj;
            if (!Objects.equals(_deadTimerValue, other.getDeadTimerValue())) {
                return false;
            }
            if (!Objects.equals(_keepAliveTimerValue, other.getKeepAliveTimerValue())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                PcepSessionConfigImpl otherImpl = (PcepSessionConfigImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<PcepSessionConfig>>, Augmentation<PcepSessionConfig>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("PcepSessionConfig");
            CodeHelpers.appendValue(helper, "_deadTimerValue", _deadTimerValue);
            CodeHelpers.appendValue(helper, "_keepAliveTimerValue", _keepAliveTimerValue);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
