package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.topology.sr.rev130819.sr.node.attributes;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Deprecated;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.topology.sr.rev130819.SegmentId;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.topology.sr.rev130819.sr.node.attributes.segments.SegmentSpecification;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link SegmentsBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     SegmentsBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new SegmentsBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of SegmentsBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see SegmentsBuilder
 * @see Builder
 *
 */
public class SegmentsBuilder implements Builder<Segments> {

    private SegmentId _segmentId;
    private SegmentSpecification _segmentSpecification;
    private SegmentsKey key;


    Map<Class<? extends Augmentation<Segments>>, Augmentation<Segments>> augmentation = Collections.emptyMap();

    public SegmentsBuilder() {
    }

    public SegmentsBuilder(Segments base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<Segments>>, Augmentation<Segments>> aug =((AugmentationHolder<Segments>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this.key = base.key();
        this._segmentId = base.getSegmentId();
        this._segmentSpecification = base.getSegmentSpecification();
    }


    public SegmentsKey key() {
        return key;
    }
    
    public SegmentId getSegmentId() {
        return _segmentId;
    }
    
    public SegmentSpecification getSegmentSpecification() {
        return _segmentSpecification;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<Segments>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(Objects.requireNonNull(augmentationType));
    }

    public SegmentsBuilder withKey(final SegmentsKey key) {
        this.key = key;
        return this;
    }
    
    public SegmentsBuilder setSegmentId(final SegmentId value) {
        this._segmentId = value;
        return this;
    }
    
    public SegmentsBuilder setSegmentSpecification(final SegmentSpecification value) {
        this._segmentSpecification = value;
        return this;
    }
    
    /**
      * Add an augmentation to this builder's product.
      *
      * @param augmentation augmentation to be added
      * @return this builder
      * @throws NullPointerException if {@code augmentation} is null
      */
    public SegmentsBuilder addAugmentation(Augmentation<Segments> augmentation) {
        return doAddAugmentation(augmentation.implementedInterface(), augmentation);
    }
    
    /**
      * Add or remove an augmentation to this builder's product.
      *
      * @param augmentationType augmentation type to be added or removed
      * @param augmentationValue augmentation value, null if the augmentation type should be removed
      * @return this builder
      * @deprecated Use either {@link #addAugmentation(Augmentation)} or {@link #removeAugmentation(Class)} instead.
      */
    @Deprecated(forRemoval = true)
    public SegmentsBuilder addAugmentation(Class<? extends Augmentation<Segments>> augmentationType, Augmentation<Segments> augmentationValue) {
        return augmentationValue == null ? removeAugmentation(augmentationType) : doAddAugmentation(augmentationType, augmentationValue);
    }
    
    /**
      * Remove an augmentation from this builder's product. If this builder does not track such an augmentation
      * type, this method does nothing.
      *
      * @param augmentationType augmentation type to be removed
      * @return this builder
      */
    public SegmentsBuilder removeAugmentation(Class<? extends Augmentation<Segments>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }
    
    private SegmentsBuilder doAddAugmentation(Class<? extends Augmentation<Segments>> augmentationType, Augmentation<Segments> augmentationValue) {
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }

    @Override
    public Segments build() {
        return new SegmentsImpl(this);
    }

    private static final class SegmentsImpl
        extends AbstractAugmentable<Segments>
        implements Segments {
    
        private final SegmentId _segmentId;
        private final SegmentSpecification _segmentSpecification;
        private final SegmentsKey key;
    
        SegmentsImpl(SegmentsBuilder base) {
            super(base.augmentation);
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new SegmentsKey(base.getSegmentId());
            }
            this._segmentId = key.getSegmentId();
            this._segmentSpecification = base.getSegmentSpecification();
        }
    
        @Override
        public SegmentsKey key() {
            return key;
        }
        
        @Override
        public SegmentId getSegmentId() {
            return _segmentId;
        }
        
        @Override
        public SegmentSpecification getSegmentSpecification() {
            return _segmentSpecification;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_segmentId);
            result = prime * result + Objects.hashCode(_segmentSpecification);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!Segments.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            Segments other = (Segments)obj;
            if (!Objects.equals(_segmentId, other.getSegmentId())) {
                return false;
            }
            if (!Objects.equals(_segmentSpecification, other.getSegmentSpecification())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                SegmentsImpl otherImpl = (SegmentsImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<Segments>>, Augmentation<Segments>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("Segments");
            CodeHelpers.appendValue(helper, "_segmentId", _segmentId);
            CodeHelpers.appendValue(helper, "_segmentSpecification", _segmentSpecification);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
