package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.dsbenchmark.rev150105.test.exec;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Integer;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.dsbenchmark.rev150105.test.exec.outer.list.InnerList;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.dsbenchmark.rev150105.test.exec.outer.list.OuterChoice;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link OuterListBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     OuterListBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new OuterListBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of OuterListBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see OuterListBuilder
 * @see Builder
 *
 */
public class OuterListBuilder implements Builder<OuterList> {

    private Integer _id;
    private List<InnerList> _innerList;
    private OuterChoice _outerChoice;
    private OuterListKey key;


    Map<Class<? extends Augmentation<OuterList>>, Augmentation<OuterList>> augmentation = Collections.emptyMap();

    public OuterListBuilder() {
    }

    public OuterListBuilder(OuterList base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<OuterList>>, Augmentation<OuterList>> aug =((AugmentationHolder<OuterList>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this.key = base.key();
        this._id = base.getId();
        this._innerList = base.getInnerList();
        this._outerChoice = base.getOuterChoice();
    }


    public OuterListKey key() {
        return key;
    }
    
    public Integer getId() {
        return _id;
    }
    
    public List<InnerList> getInnerList() {
        return _innerList;
    }
    
    public OuterChoice getOuterChoice() {
        return _outerChoice;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<OuterList>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    public OuterListBuilder withKey(final OuterListKey key) {
        this.key = key;
        return this;
    }
    
    public OuterListBuilder setId(final Integer value) {
        this._id = value;
        return this;
    }
    public OuterListBuilder setInnerList(final List<InnerList> values) {
        this._innerList = values;
        return this;
    }
    
    
    public OuterListBuilder setOuterChoice(final OuterChoice value) {
        this._outerChoice = value;
        return this;
    }
    
    public OuterListBuilder addAugmentation(Class<? extends Augmentation<OuterList>> augmentationType, Augmentation<OuterList> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public OuterListBuilder removeAugmentation(Class<? extends Augmentation<OuterList>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public OuterList build() {
        return new OuterListImpl(this);
    }

    private static final class OuterListImpl
        extends AbstractAugmentable<OuterList>
        implements OuterList {
    
        private final Integer _id;
        private final List<InnerList> _innerList;
        private final OuterChoice _outerChoice;
        private final OuterListKey key;
    
        OuterListImpl(OuterListBuilder base) {
            super(base.augmentation);
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new OuterListKey(base.getId());
            }
            this._id = key.getId();
            this._innerList = base.getInnerList();
            this._outerChoice = base.getOuterChoice();
        }
    
        @Override
        public OuterListKey key() {
            return key;
        }
        
        @Override
        public Integer getId() {
            return _id;
        }
        
        @Override
        public List<InnerList> getInnerList() {
            return _innerList;
        }
        
        @Override
        public OuterChoice getOuterChoice() {
            return _outerChoice;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_id);
            result = prime * result + Objects.hashCode(_innerList);
            result = prime * result + Objects.hashCode(_outerChoice);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!OuterList.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            OuterList other = (OuterList)obj;
            if (!Objects.equals(_id, other.getId())) {
                return false;
            }
            if (!Objects.equals(_innerList, other.getInnerList())) {
                return false;
            }
            if (!Objects.equals(_outerChoice, other.getOuterChoice())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                OuterListImpl otherImpl = (OuterListImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<OuterList>>, Augmentation<OuterList>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("OuterList");
            CodeHelpers.appendValue(helper, "_id", _id);
            CodeHelpers.appendValue(helper, "_innerList", _innerList);
            CodeHelpers.appendValue(helper, "_outerChoice", _outerChoice);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
