package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.ntfbenchmark.rev150105;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMap.Builder;
import java.lang.Class;
import java.lang.Integer;
import java.lang.Override;
import java.lang.String;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.eclipse.jdt.annotation.NonNull;
import org.eclipse.jdt.annotation.Nullable;
import org.opendaylight.yangtools.yang.binding.Augmentable;
import org.opendaylight.yangtools.yang.binding.Enumeration;
import org.opendaylight.yangtools.yang.binding.RpcInput;
import org.opendaylight.yangtools.yang.common.QName;
import org.opendaylight.yangtools.yang.common.Uint32;

/**
 *
 * <p>
 * This class represents the following YANG schema fragment defined in module <b>ntfbenchmark</b>
 * <pre>
 * input input {
 *   leaf producer-type {
 *     type enumeration {
 *       enum BLOCKING {
 *         value 1;
 *       }
 *       enum DROPPING {
 *         value 2;
 *       }
 *     }
 *   }
 *   leaf producers {
 *     type uint32;
 *     default 1;
 *   }
 *   leaf listeners {
 *     type uint32;
 *     default 1;
 *   }
 *   leaf payload-size {
 *     type uint32;
 *     default 1;
 *   }
 *   leaf iterations {
 *     type uint32;
 *     default 1;
 *   }
 * }
 * </pre>The schema path to identify an instance is
 * <i>ntfbenchmark/start-test/input</i>
 *
 * <p>To create instances of this class use {@link StartTestInputBuilder}.
 * @see StartTestInputBuilder
 *
 */
public interface StartTestInput
    extends
    RpcInput,
    Augmentable<StartTestInput>
{


    public enum ProducerType implements Enumeration {
        /**
         * The producer waits for a free slot in RPC Broker's ring buffer
         *
         */
        BLOCKING(1, "BLOCKING"),
        
        /**
         * The producer drops a notification if there is no free slot in RPC Broker's ring
         * buffer
         *
         */
        DROPPING(2, "DROPPING")
        ;
    
        private static final Map<String, ProducerType> NAME_MAP;
        private static final Map<Integer, ProducerType> VALUE_MAP;
    
        static {
            final Builder<String, ProducerType> nb = ImmutableMap.builder();
            final Builder<Integer, ProducerType> vb = ImmutableMap.builder();
            for (ProducerType enumItem : ProducerType.values()) {
                vb.put(enumItem.value, enumItem);
                nb.put(enumItem.name, enumItem);
            }
    
            NAME_MAP = nb.build();
            VALUE_MAP = vb.build();
        }
    
        private final String name;
        private final int value;
    
        private ProducerType(int value, String name) {
            this.value = value;
            this.name = name;
        }
    
        @Override
        public String getName() {
            return name;
        }
    
        @Override
        public int getIntValue() {
            return value;
        }
    
        /**
         * Return the enumeration member whose {@link #getName()} matches specified value.
         *
         * @param name YANG assigned name
         * @return corresponding ProducerType item, if present
         * @throws NullPointerException if name is null
         */
        public static Optional<ProducerType> forName(String name) {
            return Optional.ofNullable(NAME_MAP.get(Objects.requireNonNull(name)));
        }
    
        /**
         * Return the enumeration member whose {@link #getIntValue()} matches specified value.
         *
         * @param intValue integer value
         * @return corresponding ProducerType item, or null if no such item exists
         */
        public static ProducerType forValue(int intValue) {
            return VALUE_MAP.get(intValue);
        }
    }

    public static final @NonNull QName QNAME = $YangModuleInfoImpl.qnameOf("input");

    @Override
    default Class<org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.ntfbenchmark.rev150105.StartTestInput> implementedInterface() {
        return org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.ntfbenchmark.rev150105.StartTestInput.class;
    }
    
    /**
     * RPC type and client type to use in the test
     *
     *
     *
     * @return <code>org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.ntfbenchmark.rev150105.StartTestInput.ProducerType</code> <code>producerType</code>, or <code>null</code> if not present
     */
    @Nullable ProducerType getProducerType();
    
    /**
     * Number of notification producers (test client threads) to start
     *
     *
     *
     * @return <code>org.opendaylight.yangtools.yang.common.Uint32</code> <code>producers</code>, or <code>null</code> if not present
     */
    @Nullable Uint32 getProducers();
    
    /**
     * Number of notification listener instances
     *
     *
     *
     * @return <code>org.opendaylight.yangtools.yang.common.Uint32</code> <code>listeners</code>, or <code>null</code> if not present
     */
    @Nullable Uint32 getListeners();
    
    /**
     * Notification payload size: number of elements in the list of integers that is
     * the notification payload
     *
     *
     *
     * @return <code>org.opendaylight.yangtools.yang.common.Uint32</code> <code>payloadSize</code>, or <code>null</code> if not present
     */
    @Nullable Uint32 getPayloadSize();
    
    /**
     * Number of notifications to generate in each client thread
     *
     *
     *
     * @return <code>org.opendaylight.yangtools.yang.common.Uint32</code> <code>iterations</code>, or <code>null</code> if not present
     */
    @Nullable Uint32 getIterations();

}

