package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.ntfbenchmark.rev150105;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Long;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;
import org.opendaylight.yangtools.yang.common.Uint32;

/**
 * Class that builds {@link StartTestInputBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     StartTestInputBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new StartTestInputBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of StartTestInputBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see StartTestInputBuilder
 * @see Builder
 *
 */
public class StartTestInputBuilder implements Builder<StartTestInput> {

    private Uint32 _iterations;
    private Uint32 _listeners;
    private Uint32 _payloadSize;
    private StartTestInput.ProducerType _producerType;
    private Uint32 _producers;


    Map<Class<? extends Augmentation<StartTestInput>>, Augmentation<StartTestInput>> augmentation = Collections.emptyMap();

    public StartTestInputBuilder() {
    }

    public StartTestInputBuilder(StartTestInput base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<StartTestInput>>, Augmentation<StartTestInput>> aug =((AugmentationHolder<StartTestInput>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._iterations = base.getIterations();
        this._listeners = base.getListeners();
        this._payloadSize = base.getPayloadSize();
        this._producerType = base.getProducerType();
        this._producers = base.getProducers();
    }


    public Uint32 getIterations() {
        return _iterations;
    }
    
    public Uint32 getListeners() {
        return _listeners;
    }
    
    public Uint32 getPayloadSize() {
        return _payloadSize;
    }
    
    public StartTestInput.ProducerType getProducerType() {
        return _producerType;
    }
    
    public Uint32 getProducers() {
        return _producers;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<StartTestInput>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    
    public StartTestInputBuilder setIterations(final Uint32 value) {
        this._iterations = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setIterations(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public StartTestInputBuilder setIterations(final Long value) {
        return setIterations(CodeHelpers.compatUint(value));
    }
    
    public StartTestInputBuilder setListeners(final Uint32 value) {
        this._listeners = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setListeners(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public StartTestInputBuilder setListeners(final Long value) {
        return setListeners(CodeHelpers.compatUint(value));
    }
    
    public StartTestInputBuilder setPayloadSize(final Uint32 value) {
        this._payloadSize = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setPayloadSize(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public StartTestInputBuilder setPayloadSize(final Long value) {
        return setPayloadSize(CodeHelpers.compatUint(value));
    }
    
    public StartTestInputBuilder setProducerType(final StartTestInput.ProducerType value) {
        this._producerType = value;
        return this;
    }
    
    public StartTestInputBuilder setProducers(final Uint32 value) {
        this._producers = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setProducers(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public StartTestInputBuilder setProducers(final Long value) {
        return setProducers(CodeHelpers.compatUint(value));
    }
    
    public StartTestInputBuilder addAugmentation(Class<? extends Augmentation<StartTestInput>> augmentationType, Augmentation<StartTestInput> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public StartTestInputBuilder removeAugmentation(Class<? extends Augmentation<StartTestInput>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public StartTestInput build() {
        return new StartTestInputImpl(this);
    }

    private static final class StartTestInputImpl
        extends AbstractAugmentable<StartTestInput>
        implements StartTestInput {
    
        private final Uint32 _iterations;
        private final Uint32 _listeners;
        private final Uint32 _payloadSize;
        private final StartTestInput.ProducerType _producerType;
        private final Uint32 _producers;
    
        StartTestInputImpl(StartTestInputBuilder base) {
            super(base.augmentation);
            this._iterations = base.getIterations();
            this._listeners = base.getListeners();
            this._payloadSize = base.getPayloadSize();
            this._producerType = base.getProducerType();
            this._producers = base.getProducers();
        }
    
        @Override
        public Uint32 getIterations() {
            return _iterations;
        }
        
        @Override
        public Uint32 getListeners() {
            return _listeners;
        }
        
        @Override
        public Uint32 getPayloadSize() {
            return _payloadSize;
        }
        
        @Override
        public StartTestInput.ProducerType getProducerType() {
            return _producerType;
        }
        
        @Override
        public Uint32 getProducers() {
            return _producers;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_iterations);
            result = prime * result + Objects.hashCode(_listeners);
            result = prime * result + Objects.hashCode(_payloadSize);
            result = prime * result + Objects.hashCode(_producerType);
            result = prime * result + Objects.hashCode(_producers);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!StartTestInput.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            StartTestInput other = (StartTestInput)obj;
            if (!Objects.equals(_iterations, other.getIterations())) {
                return false;
            }
            if (!Objects.equals(_listeners, other.getListeners())) {
                return false;
            }
            if (!Objects.equals(_payloadSize, other.getPayloadSize())) {
                return false;
            }
            if (!Objects.equals(_producerType, other.getProducerType())) {
                return false;
            }
            if (!Objects.equals(_producers, other.getProducers())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                StartTestInputImpl otherImpl = (StartTestInputImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<StartTestInput>>, Augmentation<StartTestInput>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("StartTestInput");
            CodeHelpers.appendValue(helper, "_iterations", _iterations);
            CodeHelpers.appendValue(helper, "_listeners", _listeners);
            CodeHelpers.appendValue(helper, "_payloadSize", _payloadSize);
            CodeHelpers.appendValue(helper, "_producerType", _producerType);
            CodeHelpers.appendValue(helper, "_producers", _producers);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
