package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.rpcbenchmark.rev150702;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Long;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;
import org.opendaylight.yangtools.yang.common.Uint32;

/**
 * Class that builds {@link StartTestInputBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     StartTestInputBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new StartTestInputBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of StartTestInputBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see StartTestInputBuilder
 * @see Builder
 *
 */
public class StartTestInputBuilder implements Builder<StartTestInput> {

    private Uint32 _iterations;
    private Uint32 _numClients;
    private Uint32 _numServers;
    private StartTestInput.Operation _operation;
    private Uint32 _payloadSize;


    Map<Class<? extends Augmentation<StartTestInput>>, Augmentation<StartTestInput>> augmentation = Collections.emptyMap();

    public StartTestInputBuilder() {
    }

    public StartTestInputBuilder(StartTestInput base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<StartTestInput>>, Augmentation<StartTestInput>> aug =((AugmentationHolder<StartTestInput>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._iterations = base.getIterations();
        this._numClients = base.getNumClients();
        this._numServers = base.getNumServers();
        this._operation = base.getOperation();
        this._payloadSize = base.getPayloadSize();
    }


    public Uint32 getIterations() {
        return _iterations;
    }
    
    public Uint32 getNumClients() {
        return _numClients;
    }
    
    public Uint32 getNumServers() {
        return _numServers;
    }
    
    public StartTestInput.Operation getOperation() {
        return _operation;
    }
    
    public Uint32 getPayloadSize() {
        return _payloadSize;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<StartTestInput>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    
    public StartTestInputBuilder setIterations(final Uint32 value) {
        this._iterations = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setIterations(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public StartTestInputBuilder setIterations(final Long value) {
        return setIterations(CodeHelpers.compatUint(value));
    }
    
    public StartTestInputBuilder setNumClients(final Uint32 value) {
        this._numClients = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setNumClients(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public StartTestInputBuilder setNumClients(final Long value) {
        return setNumClients(CodeHelpers.compatUint(value));
    }
    
    public StartTestInputBuilder setNumServers(final Uint32 value) {
        this._numServers = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setNumServers(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public StartTestInputBuilder setNumServers(final Long value) {
        return setNumServers(CodeHelpers.compatUint(value));
    }
    
    public StartTestInputBuilder setOperation(final StartTestInput.Operation value) {
        this._operation = value;
        return this;
    }
    
    public StartTestInputBuilder setPayloadSize(final Uint32 value) {
        this._payloadSize = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setPayloadSize(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public StartTestInputBuilder setPayloadSize(final Long value) {
        return setPayloadSize(CodeHelpers.compatUint(value));
    }
    
    public StartTestInputBuilder addAugmentation(Class<? extends Augmentation<StartTestInput>> augmentationType, Augmentation<StartTestInput> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public StartTestInputBuilder removeAugmentation(Class<? extends Augmentation<StartTestInput>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public StartTestInput build() {
        return new StartTestInputImpl(this);
    }

    private static final class StartTestInputImpl
        extends AbstractAugmentable<StartTestInput>
        implements StartTestInput {
    
        private final Uint32 _iterations;
        private final Uint32 _numClients;
        private final Uint32 _numServers;
        private final StartTestInput.Operation _operation;
        private final Uint32 _payloadSize;
    
        StartTestInputImpl(StartTestInputBuilder base) {
            super(base.augmentation);
            this._iterations = base.getIterations();
            this._numClients = base.getNumClients();
            this._numServers = base.getNumServers();
            this._operation = base.getOperation();
            this._payloadSize = base.getPayloadSize();
        }
    
        @Override
        public Uint32 getIterations() {
            return _iterations;
        }
        
        @Override
        public Uint32 getNumClients() {
            return _numClients;
        }
        
        @Override
        public Uint32 getNumServers() {
            return _numServers;
        }
        
        @Override
        public StartTestInput.Operation getOperation() {
            return _operation;
        }
        
        @Override
        public Uint32 getPayloadSize() {
            return _payloadSize;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_iterations);
            result = prime * result + Objects.hashCode(_numClients);
            result = prime * result + Objects.hashCode(_numServers);
            result = prime * result + Objects.hashCode(_operation);
            result = prime * result + Objects.hashCode(_payloadSize);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!StartTestInput.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            StartTestInput other = (StartTestInput)obj;
            if (!Objects.equals(_iterations, other.getIterations())) {
                return false;
            }
            if (!Objects.equals(_numClients, other.getNumClients())) {
                return false;
            }
            if (!Objects.equals(_numServers, other.getNumServers())) {
                return false;
            }
            if (!Objects.equals(_operation, other.getOperation())) {
                return false;
            }
            if (!Objects.equals(_payloadSize, other.getPayloadSize())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                StartTestInputImpl otherImpl = (StartTestInputImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<StartTestInput>>, Augmentation<StartTestInput>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("StartTestInput");
            CodeHelpers.appendValue(helper, "_iterations", _iterations);
            CodeHelpers.appendValue(helper, "_numClients", _numClients);
            CodeHelpers.appendValue(helper, "_numServers", _numServers);
            CodeHelpers.appendValue(helper, "_operation", _operation);
            CodeHelpers.appendValue(helper, "_payloadSize", _payloadSize);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
