package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.rpcbenchmark.rev150702;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Long;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;
import org.opendaylight.yangtools.yang.common.Uint32;

/**
 * Class that builds {@link TestStatusOutputBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     TestStatusOutputBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new TestStatusOutputBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of TestStatusOutputBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see TestStatusOutputBuilder
 * @see Builder
 *
 */
public class TestStatusOutputBuilder implements Builder<TestStatusOutput> {

    private TestStatusOutput.ExecStatus _execStatus;
    private Uint32 _globalServerCnt;


    Map<Class<? extends Augmentation<TestStatusOutput>>, Augmentation<TestStatusOutput>> augmentation = Collections.emptyMap();

    public TestStatusOutputBuilder() {
    }

    public TestStatusOutputBuilder(TestStatusOutput base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<TestStatusOutput>>, Augmentation<TestStatusOutput>> aug =((AugmentationHolder<TestStatusOutput>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._execStatus = base.getExecStatus();
        this._globalServerCnt = base.getGlobalServerCnt();
    }


    public TestStatusOutput.ExecStatus getExecStatus() {
        return _execStatus;
    }
    
    public Uint32 getGlobalServerCnt() {
        return _globalServerCnt;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<TestStatusOutput>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    
    public TestStatusOutputBuilder setExecStatus(final TestStatusOutput.ExecStatus value) {
        this._execStatus = value;
        return this;
    }
    
    public TestStatusOutputBuilder setGlobalServerCnt(final Uint32 value) {
        this._globalServerCnt = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setGlobalServerCnt(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public TestStatusOutputBuilder setGlobalServerCnt(final Long value) {
        return setGlobalServerCnt(CodeHelpers.compatUint(value));
    }
    
    public TestStatusOutputBuilder addAugmentation(Class<? extends Augmentation<TestStatusOutput>> augmentationType, Augmentation<TestStatusOutput> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public TestStatusOutputBuilder removeAugmentation(Class<? extends Augmentation<TestStatusOutput>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public TestStatusOutput build() {
        return new TestStatusOutputImpl(this);
    }

    private static final class TestStatusOutputImpl
        extends AbstractAugmentable<TestStatusOutput>
        implements TestStatusOutput {
    
        private final TestStatusOutput.ExecStatus _execStatus;
        private final Uint32 _globalServerCnt;
    
        TestStatusOutputImpl(TestStatusOutputBuilder base) {
            super(base.augmentation);
            this._execStatus = base.getExecStatus();
            this._globalServerCnt = base.getGlobalServerCnt();
        }
    
        @Override
        public TestStatusOutput.ExecStatus getExecStatus() {
            return _execStatus;
        }
        
        @Override
        public Uint32 getGlobalServerCnt() {
            return _globalServerCnt;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_execStatus);
            result = prime * result + Objects.hashCode(_globalServerCnt);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!TestStatusOutput.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            TestStatusOutput other = (TestStatusOutput)obj;
            if (!Objects.equals(_execStatus, other.getExecStatus())) {
                return false;
            }
            if (!Objects.equals(_globalServerCnt, other.getGlobalServerCnt())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                TestStatusOutputImpl otherImpl = (TestStatusOutputImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<TestStatusOutput>>, Augmentation<TestStatusOutput>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("TestStatusOutput");
            CodeHelpers.appendValue(helper, "_execStatus", _execStatus);
            CodeHelpers.appendValue(helper, "_globalServerCnt", _globalServerCnt);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
