package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.rpcbenchmark.rev150702;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMap.Builder;
import java.lang.Class;
import java.lang.Integer;
import java.lang.Override;
import java.lang.String;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.eclipse.jdt.annotation.NonNull;
import org.eclipse.jdt.annotation.Nullable;
import org.opendaylight.yangtools.yang.binding.Augmentable;
import org.opendaylight.yangtools.yang.binding.Enumeration;
import org.opendaylight.yangtools.yang.binding.RpcInput;
import org.opendaylight.yangtools.yang.common.QName;
import org.opendaylight.yangtools.yang.common.Uint32;

/**
 *
 * <p>
 * This class represents the following YANG schema fragment defined in module <b>rpcbenchmark</b>
 * <pre>
 * input input {
 *   leaf operation {
 *     type enumeration {
 *       enum GLOBAL-RTC {
 *         value 1;
 *       }
 *       enum ROUTED-RTC {
 *         value 2;
 *       }
 *     }
 *   }
 *   leaf num-clients {
 *     type uint32;
 *     default 1;
 *   }
 *   leaf num-servers {
 *     type uint32;
 *     default 1;
 *   }
 *   leaf payload-size {
 *     type uint32;
 *     default 1;
 *   }
 *   leaf iterations {
 *     type uint32;
 *     default 1;
 *   }
 * }
 * </pre>The schema path to identify an instance is
 * <i>rpcbenchmark/start-test/input</i>
 *
 * <p>To create instances of this class use {@link StartTestInputBuilder}.
 * @see StartTestInputBuilder
 *
 */
public interface StartTestInput
    extends
    RpcInput,
    Augmentable<StartTestInput>
{


    public enum Operation implements Enumeration {
        /**
         * Use Global RPC service and run-to-completion client
         *
         */
        GLOBALRTC(1, "GLOBAL-RTC"),
        
        /**
         * Use routed RPC service and run-to-completion client. RPC server instances are
         * dynamically created when the test starts and deleted when the test finishes
         *
         */
        ROUTEDRTC(2, "ROUTED-RTC")
        ;
    
        private static final Map<String, Operation> NAME_MAP;
        private static final Map<Integer, Operation> VALUE_MAP;
    
        static {
            final Builder<String, Operation> nb = ImmutableMap.builder();
            final Builder<Integer, Operation> vb = ImmutableMap.builder();
            for (Operation enumItem : Operation.values()) {
                vb.put(enumItem.value, enumItem);
                nb.put(enumItem.name, enumItem);
            }
    
            NAME_MAP = nb.build();
            VALUE_MAP = vb.build();
        }
    
        private final String name;
        private final int value;
    
        private Operation(int value, String name) {
            this.value = value;
            this.name = name;
        }
    
        @Override
        public String getName() {
            return name;
        }
    
        @Override
        public int getIntValue() {
            return value;
        }
    
        /**
         * Return the enumeration member whose {@link #getName()} matches specified value.
         *
         * @param name YANG assigned name
         * @return corresponding Operation item, if present
         * @throws NullPointerException if name is null
         */
        public static Optional<Operation> forName(String name) {
            return Optional.ofNullable(NAME_MAP.get(Objects.requireNonNull(name)));
        }
    
        /**
         * Return the enumeration member whose {@link #getIntValue()} matches specified value.
         *
         * @param intValue integer value
         * @return corresponding Operation item, or null if no such item exists
         */
        public static Operation forValue(int intValue) {
            return VALUE_MAP.get(intValue);
        }
    }

    public static final @NonNull QName QNAME = $YangModuleInfoImpl.qnameOf("input");

    @Override
    default Class<org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.rpcbenchmark.rev150702.StartTestInput> implementedInterface() {
        return org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.rpcbenchmark.rev150702.StartTestInput.class;
    }
    
    /**
     * RPC type and client type to use in the test
     *
     *
     *
     * @return <code>org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.rpcbenchmark.rev150702.StartTestInput.Operation</code> <code>operation</code>, or <code>null</code> if not present
     */
    @Nullable Operation getOperation();
    
    /**
     * Number of clients (test client threads) to start
     *
     *
     *
     * @return <code>org.opendaylight.yangtools.yang.common.Uint32</code> <code>numClients</code>, or <code>null</code> if not present
     */
    @Nullable Uint32 getNumClients();
    
    /**
     * Number of RPC server instances. Only valid for routed RPCs.
     *
     *
     *
     * @return <code>org.opendaylight.yangtools.yang.common.Uint32</code> <code>numServers</code>, or <code>null</code> if not present
     */
    @Nullable Uint32 getNumServers();
    
    /**
     * Input/Output payload size: number of elements in the list of integers that is
     * the input and output RPC payload
     *
     *
     *
     * @return <code>org.opendaylight.yangtools.yang.common.Uint32</code> <code>payloadSize</code>, or <code>null</code> if not present
     */
    @Nullable Uint32 getPayloadSize();
    
    /**
     * Number of calls to the specified RPC server that is to be made by each client
     *
     *
     *
     * @return <code>org.opendaylight.yangtools.yang.common.Uint32</code> <code>iterations</code>, or <code>null</code> if not present
     */
    @Nullable Uint32 getIterations();

}

