package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.rpcbenchmark.rev150702;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Deprecated;
import java.lang.Long;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;
import org.opendaylight.yangtools.yang.common.Uint32;

/**
 * Class that builds {@link StartTestOutputBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     StartTestOutputBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new StartTestOutputBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of StartTestOutputBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see StartTestOutputBuilder
 * @see Builder
 *
 */
public class StartTestOutputBuilder implements Builder<StartTestOutput> {

    private Uint32 _execTime;
    private Uint32 _globalRtcClientError;
    private Uint32 _globalRtcClientOk;
    private Uint32 _rate;


    Map<Class<? extends Augmentation<StartTestOutput>>, Augmentation<StartTestOutput>> augmentation = Collections.emptyMap();

    public StartTestOutputBuilder() {
    }

    public StartTestOutputBuilder(StartTestOutput base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<StartTestOutput>>, Augmentation<StartTestOutput>> aug =((AugmentationHolder<StartTestOutput>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._execTime = base.getExecTime();
        this._globalRtcClientError = base.getGlobalRtcClientError();
        this._globalRtcClientOk = base.getGlobalRtcClientOk();
        this._rate = base.getRate();
    }


    public Uint32 getExecTime() {
        return _execTime;
    }
    
    public Uint32 getGlobalRtcClientError() {
        return _globalRtcClientError;
    }
    
    public Uint32 getGlobalRtcClientOk() {
        return _globalRtcClientOk;
    }
    
    public Uint32 getRate() {
        return _rate;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<StartTestOutput>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(Objects.requireNonNull(augmentationType));
    }

    
    public StartTestOutputBuilder setExecTime(final Uint32 value) {
        this._execTime = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setExecTime(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public StartTestOutputBuilder setExecTime(final Long value) {
        return setExecTime(CodeHelpers.compatUint(value));
    }
    
    public StartTestOutputBuilder setGlobalRtcClientError(final Uint32 value) {
        this._globalRtcClientError = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setGlobalRtcClientError(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public StartTestOutputBuilder setGlobalRtcClientError(final Long value) {
        return setGlobalRtcClientError(CodeHelpers.compatUint(value));
    }
    
    public StartTestOutputBuilder setGlobalRtcClientOk(final Uint32 value) {
        this._globalRtcClientOk = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setGlobalRtcClientOk(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public StartTestOutputBuilder setGlobalRtcClientOk(final Long value) {
        return setGlobalRtcClientOk(CodeHelpers.compatUint(value));
    }
    
    public StartTestOutputBuilder setRate(final Uint32 value) {
        this._rate = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setRate(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public StartTestOutputBuilder setRate(final Long value) {
        return setRate(CodeHelpers.compatUint(value));
    }
    
    /**
      * Add an augmentation to this builder's product.
      *
      * @param augmentation augmentation to be added
      * @return this builder
      * @throws NullPointerException if {@code augmentation} is null
      */
    public StartTestOutputBuilder addAugmentation(Augmentation<StartTestOutput> augmentation) {
        return doAddAugmentation(augmentation.implementedInterface(), augmentation);
    }
    
    /**
      * Add or remove an augmentation to this builder's product.
      *
      * @param augmentationType augmentation type to be added or removed
      * @param augmentationValue augmentation value, null if the augmentation type should be removed
      * @return this builder
      * @deprecated Use either {@link #addAugmentation(Augmentation)} or {@link #removeAugmentation(Class)} instead.
      */
    @Deprecated(forRemoval = true)
    public StartTestOutputBuilder addAugmentation(Class<? extends Augmentation<StartTestOutput>> augmentationType, Augmentation<StartTestOutput> augmentationValue) {
        return augmentationValue == null ? removeAugmentation(augmentationType) : doAddAugmentation(augmentationType, augmentationValue);
    }
    
    /**
      * Remove an augmentation from this builder's product. If this builder does not track such an augmentation
      * type, this method does nothing.
      *
      * @param augmentationType augmentation type to be removed
      * @return this builder
      */
    public StartTestOutputBuilder removeAugmentation(Class<? extends Augmentation<StartTestOutput>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }
    
    private StartTestOutputBuilder doAddAugmentation(Class<? extends Augmentation<StartTestOutput>> augmentationType, Augmentation<StartTestOutput> augmentationValue) {
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }

    @Override
    public StartTestOutput build() {
        return new StartTestOutputImpl(this);
    }

    private static final class StartTestOutputImpl
        extends AbstractAugmentable<StartTestOutput>
        implements StartTestOutput {
    
        private final Uint32 _execTime;
        private final Uint32 _globalRtcClientError;
        private final Uint32 _globalRtcClientOk;
        private final Uint32 _rate;
    
        StartTestOutputImpl(StartTestOutputBuilder base) {
            super(base.augmentation);
            this._execTime = base.getExecTime();
            this._globalRtcClientError = base.getGlobalRtcClientError();
            this._globalRtcClientOk = base.getGlobalRtcClientOk();
            this._rate = base.getRate();
        }
    
        @Override
        public Uint32 getExecTime() {
            return _execTime;
        }
        
        @Override
        public Uint32 getGlobalRtcClientError() {
            return _globalRtcClientError;
        }
        
        @Override
        public Uint32 getGlobalRtcClientOk() {
            return _globalRtcClientOk;
        }
        
        @Override
        public Uint32 getRate() {
            return _rate;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_execTime);
            result = prime * result + Objects.hashCode(_globalRtcClientError);
            result = prime * result + Objects.hashCode(_globalRtcClientOk);
            result = prime * result + Objects.hashCode(_rate);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!StartTestOutput.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            StartTestOutput other = (StartTestOutput)obj;
            if (!Objects.equals(_execTime, other.getExecTime())) {
                return false;
            }
            if (!Objects.equals(_globalRtcClientError, other.getGlobalRtcClientError())) {
                return false;
            }
            if (!Objects.equals(_globalRtcClientOk, other.getGlobalRtcClientOk())) {
                return false;
            }
            if (!Objects.equals(_rate, other.getRate())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                StartTestOutputImpl otherImpl = (StartTestOutputImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<StartTestOutput>>, Augmentation<StartTestOutput>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("StartTestOutput");
            CodeHelpers.appendValue(helper, "_execTime", _execTime);
            CodeHelpers.appendValue(helper, "_globalRtcClientError", _globalRtcClientError);
            CodeHelpers.appendValue(helper, "_globalRtcClientOk", _globalRtcClientOk);
            CodeHelpers.appendValue(helper, "_rate", _rate);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
