package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.dsbenchmark.rev150105;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMap.Builder;
import java.lang.Class;
import java.lang.Integer;
import java.lang.Long;
import java.lang.Override;
import java.lang.String;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.eclipse.jdt.annotation.NonNull;
import org.eclipse.jdt.annotation.Nullable;
import org.opendaylight.yangtools.yang.binding.Augmentable;
import org.opendaylight.yangtools.yang.binding.Enumeration;
import org.opendaylight.yangtools.yang.binding.RpcOutput;
import org.opendaylight.yangtools.yang.common.QName;
import org.opendaylight.yangtools.yang.common.Uint32;

/**
 *
 * <p>
 * This class represents the following YANG schema fragment defined in module <b>dsbenchmark</b>
 * <pre>
 * output output {
 *   leaf status {
 *     type enumeration {
 *       enum OK {
 *         value 1;
 *       }
 *       enum FAILED {
 *         value 2;
 *       }
 *       enum TEST-IN-PROGRESS {
 *         value 3;
 *       }
 *     }
 *   }
 *   leaf listBuildTime {
 *     type int64;
 *     units microseconds;
 *   }
 *   leaf execTime {
 *     type int64;
 *     units microseconds;
 *   }
 *   leaf txOk {
 *     type uint32;
 *   }
 *   leaf txError {
 *     type uint32;
 *   }
 *   leaf ntfOk {
 *     type uint32;
 *   }
 *   leaf dataChangeEventsOk {
 *     type uint32;
 *   }
 * }
 * </pre>The schema path to identify an instance is
 * <i>dsbenchmark/start-test/output</i>
 *
 * <p>To create instances of this class use {@link StartTestOutputBuilder}.
 * @see StartTestOutputBuilder
 *
 */
public interface StartTestOutput
    extends
    RpcOutput,
    Augmentable<StartTestOutput>
{


    public enum Status implements Enumeration {
        OK(1, "OK"),
        
        FAILED(2, "FAILED"),
        
        TESTINPROGRESS(3, "TEST-IN-PROGRESS")
        ;
    
        private static final Map<String, Status> NAME_MAP;
        private static final Map<Integer, Status> VALUE_MAP;
    
        static {
            final Builder<String, Status> nb = ImmutableMap.builder();
            final Builder<Integer, Status> vb = ImmutableMap.builder();
            for (Status enumItem : Status.values()) {
                vb.put(enumItem.value, enumItem);
                nb.put(enumItem.name, enumItem);
            }
    
            NAME_MAP = nb.build();
            VALUE_MAP = vb.build();
        }
    
        private final String name;
        private final int value;
    
        private Status(int value, String name) {
            this.value = value;
            this.name = name;
        }
    
        @Override
        public String getName() {
            return name;
        }
    
        @Override
        public int getIntValue() {
            return value;
        }
    
        /**
         * Return the enumeration member whose {@link #getName()} matches specified value.
         *
         * @param name YANG assigned name
         * @return corresponding Status item, if present
         * @throws NullPointerException if name is null
         */
        public static Optional<Status> forName(String name) {
            return Optional.ofNullable(NAME_MAP.get(Objects.requireNonNull(name)));
        }
    
        /**
         * Return the enumeration member whose {@link #getIntValue()} matches specified value.
         *
         * @param intValue integer value
         * @return corresponding Status item, or null if no such item exists
         */
        public static Status forValue(int intValue) {
            return VALUE_MAP.get(intValue);
        }
    }

    public static final @NonNull QName QNAME = $YangModuleInfoImpl.qnameOf("output");

    @Override
    default Class<org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.dsbenchmark.rev150105.StartTestOutput> implementedInterface() {
        return org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.dsbenchmark.rev150105.StartTestOutput.class;
    }
    
    /**
     * Indicates whether the test finished successfuly
     *
     *
     *
     * @return <code>org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.dsbenchmark.rev150105.StartTestOutput.Status</code> <code>status</code>, or <code>null</code> if not present
     */
    @Nullable Status getStatus();
    
    /**
     * The time it took to build the list of lists
     *
     *
     *
     * @return <code>java.lang.Long</code> <code>listBuildTime</code>, or <code>null</code> if not present
     */
    @Nullable Long getListBuildTime();
    
    /**
     * The time it took to execute all transactions
     *
     *
     *
     * @return <code>java.lang.Long</code> <code>execTime</code>, or <code>null</code> if not present
     */
    @Nullable Long getExecTime();
    
    /**
     * The number of successful transactions
     *
     *
     *
     * @return <code>org.opendaylight.yangtools.yang.common.Uint32</code> <code>txOk</code>, or <code>null</code> if not present
     */
    @Nullable Uint32 getTxOk();
    
    /**
     * The number of failed transactions
     *
     *
     *
     * @return <code>org.opendaylight.yangtools.yang.common.Uint32</code> <code>txError</code>, or <code>null</code> if not present
     */
    @Nullable Uint32 getTxError();
    
    /**
     * The number of successfully received data tree change notifications
     *
     *
     *
     * @return <code>org.opendaylight.yangtools.yang.common.Uint32</code> <code>ntfOk</code>, or <code>null</code> if not present
     */
    @Nullable Uint32 getNtfOk();
    
    /**
     * The number of data change events received in data tree change notifications
     *
     *
     *
     * @return <code>org.opendaylight.yangtools.yang.common.Uint32</code> <code>dataChangeEventsOk</code>, or <code>null</code> if not present
     */
    @Nullable Uint32 getDataChangeEventsOk();

}

