/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.storage.journal;

import com.google.common.base.Preconditions;
import io.atomix.storage.journal.JournalSegment;
import io.atomix.storage.journal.JournalSegmentDescriptor;
import io.atomix.storage.journal.JournalSegmentFile;
import io.atomix.storage.journal.SegmentedByteBufReader;
import io.atomix.storage.journal.SegmentedByteBufWriter;
import io.atomix.storage.journal.SegmentedCommitsByteBufReader;
import io.atomix.storage.journal.StorageException;
import io.atomix.storage.journal.StorageLevel;
import io.netty.buffer.ByteBufAllocator;
import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.function.BiFunction;
import org.eclipse.jdt.annotation.NonNull;
import org.eclipse.jdt.annotation.NonNullByDefault;
import org.eclipse.jdt.annotation.Nullable;
import org.opendaylight.controller.raft.journal.EntryReader;
import org.opendaylight.controller.raft.journal.EntryWriter;
import org.opendaylight.controller.raft.journal.RaftJournal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SegmentedByteBufJournal
implements RaftJournal {
    private static final Logger LOG = LoggerFactory.getLogger(SegmentedByteBufJournal.class);
    private static final int SEGMENT_BUFFER_FACTOR = 3;
    private final ConcurrentNavigableMap<Long, JournalSegment> segments = new ConcurrentSkipListMap<Long, JournalSegment>();
    private final Collection<EntryReader> readers = ConcurrentHashMap.newKeySet();
    private final @NonNull ByteBufAllocator allocator;
    private final @NonNull StorageLevel storageLevel;
    private final @NonNull File directory;
    private final @NonNull String name;
    private final @NonNull EntryWriter writer;
    private final int maxSegmentSize;
    private final int maxEntrySize;
    @Deprecated(forRemoval=true)
    private final int maxEntriesPerSegment;
    private final double indexDensity;
    private final boolean flushOnCommit;
    private JournalSegment currentSegment;
    private volatile long commitIndex;

    SegmentedByteBufJournal(String name, StorageLevel storageLevel, File directory, int maxSegmentSize, int maxEntrySize, int maxEntriesPerSegment, double indexDensity, boolean flushOnCommit, ByteBufAllocator allocator) {
        this.name = Objects.requireNonNull(name, "name cannot be null");
        this.storageLevel = Objects.requireNonNull(storageLevel, "storageLevel cannot be null");
        this.directory = Objects.requireNonNull(directory, "directory cannot be null");
        this.allocator = Objects.requireNonNull(allocator, "allocator cannot be null");
        this.maxSegmentSize = maxSegmentSize;
        this.maxEntrySize = maxEntrySize;
        this.maxEntriesPerSegment = maxEntriesPerSegment;
        this.indexDensity = indexDensity;
        this.flushOnCommit = flushOnCommit;
        for (JournalSegment segment : this.loadSegments()) {
            this.segments.put(segment.firstIndex(), segment);
        }
        this.currentSegment = this.ensureLastSegment();
        this.writer = new SegmentedByteBufWriter(this);
    }

    public long size() {
        return this.segments.values().stream().mapToLong(segment -> {
            try {
                return segment.file().size();
            }
            catch (IOException e) {
                throw new StorageException(e);
            }
        }).sum();
    }

    public long firstIndex() {
        return this.firstSegment().firstIndex();
    }

    public long lastIndex() {
        return this.lastSegment().lastIndex();
    }

    public EntryWriter writer() {
        return this.writer;
    }

    public EntryReader openReader(long index) {
        return this.openReader(index, SegmentedByteBufReader::new);
    }

    @NonNullByDefault
    private EntryReader openReader(long index, BiFunction<SegmentedByteBufJournal, JournalSegment, EntryReader> constructor) {
        EntryReader reader = constructor.apply(this, this.segment(index));
        reader.reset(index);
        this.readers.add(reader);
        return reader;
    }

    public EntryReader openCommitsReader(long index) {
        return this.openReader(index, SegmentedCommitsByteBufReader::new);
    }

    private void assertOpen() {
        Preconditions.checkState((this.currentSegment != null ? 1 : 0) != 0, (Object)"journal not open");
    }

    private void assertDiskSpace() {
        if (this.directory.getUsableSpace() < (long)(this.maxSegmentSize * 3)) {
            throw new StorageException.OutOfDiskSpace("Not enough space to allocate a new journal segment");
        }
    }

    JournalSegment resetSegments(long index) {
        JournalSegment newSegment;
        this.assertOpen();
        JournalSegment firstSegment = this.firstSegment();
        if (index == firstSegment.firstIndex()) {
            return firstSegment;
        }
        this.segments.values().forEach(JournalSegment::delete);
        this.segments.clear();
        this.currentSegment = newSegment = this.createInitialSegment();
        return newSegment;
    }

    JournalSegment firstSegment() {
        this.assertOpen();
        return (JournalSegment)this.segments.firstEntry().getValue();
    }

    JournalSegment lastSegment() {
        this.assertOpen();
        return (JournalSegment)this.segments.lastEntry().getValue();
    }

    @Nullable JournalSegment tryNextSegment(long index) {
        Map.Entry higherEntry = this.segments.higherEntry(index);
        return higherEntry != null ? (JournalSegment)higherEntry.getValue() : null;
    }

    synchronized @NonNull JournalSegment createNextSegment() {
        this.assertOpen();
        this.assertDiskSpace();
        long index = this.currentSegment.lastIndex() + 1L;
        JournalSegment lastSegment = this.lastSegment();
        JournalSegment nextSegment = this.createSegment(lastSegment.file().segmentId() + 1L, index);
        this.segments.put(index, nextSegment);
        this.currentSegment = nextSegment;
        return nextSegment;
    }

    synchronized JournalSegment segment(long index) {
        this.assertOpen();
        if (this.currentSegment != null && index > this.currentSegment.firstIndex()) {
            return this.currentSegment;
        }
        Map.Entry segment = this.segments.floorEntry(index);
        return segment != null ? (JournalSegment)segment.getValue() : this.firstSegment();
    }

    synchronized void removeSegment(JournalSegment segment) {
        this.segments.remove(segment.firstIndex());
        segment.delete();
        this.currentSegment = this.ensureLastSegment();
    }

    private @NonNull JournalSegment createSegment(long segmentId, long firstIndex) {
        JournalSegmentFile file;
        try {
            file = JournalSegmentFile.createNew(this.name, this.directory, this.allocator, JournalSegmentDescriptor.builder().withId(segmentId).withIndex(firstIndex).withMaxSegmentSize(this.maxSegmentSize).withMaxEntries(this.maxEntriesPerSegment).withUpdated(System.currentTimeMillis()).build());
        }
        catch (IOException e) {
            throw new StorageException(e);
        }
        JournalSegment segment = new JournalSegment(file, this.storageLevel, this.maxEntrySize, this.indexDensity);
        LOG.debug("Created segment: {}", (Object)segment);
        return segment;
    }

    private @NonNull JournalSegment createInitialSegment() {
        JournalSegment segment = this.createSegment(1L, 1L);
        this.segments.put(1L, segment);
        return segment;
    }

    private JournalSegment ensureLastSegment() {
        Map.Entry lastEntry = this.segments.lastEntry();
        return lastEntry != null ? (JournalSegment)lastEntry.getValue() : this.createInitialSegment();
    }

    private Collection<JournalSegment> loadSegments() {
        this.directory.mkdirs();
        TreeMap<Long, JournalSegment> segmentsMap = new TreeMap<Long, JournalSegment>();
        for (File file : this.directory.listFiles(File::isFile)) {
            JournalSegmentFile segmentFile;
            if (!JournalSegmentFile.isSegmentFile(this.name, file)) continue;
            try {
                segmentFile = JournalSegmentFile.openExisting(file.toPath(), this.allocator);
            }
            catch (IOException e) {
                throw new StorageException(e);
            }
            LOG.debug("Loaded disk segment: {} ({})", (Object)segmentFile.segmentId(), (Object)segmentFile.path());
            JournalSegment segment = new JournalSegment(segmentFile, this.storageLevel, this.maxEntrySize, this.indexDensity);
            segmentsMap.put(segment.firstIndex(), segment);
        }
        JournalSegment previousSegment = null;
        boolean corrupted = false;
        Iterator iterator = segmentsMap.entrySet().iterator();
        while (iterator.hasNext()) {
            JournalSegment segment = (JournalSegment)iterator.next().getValue();
            if (previousSegment != null && previousSegment.lastIndex() != segment.firstIndex() - 1L) {
                LOG.warn("Journal is inconsistent. {} is not aligned with prior segment {}", (Object)segment.file().path(), (Object)previousSegment.file().path());
                corrupted = true;
            }
            if (corrupted) {
                segment.delete();
                iterator.remove();
            }
            previousSegment = segment;
        }
        return segmentsMap.values();
    }

    void resetHead(long index) {
        for (EntryReader reader : this.readers) {
            if (reader.nextIndex() >= index) continue;
            reader.reset(index);
        }
    }

    void resetTail(long index) {
        for (EntryReader reader : this.readers) {
            if (reader.nextIndex() < index) continue;
            reader.reset(index);
        }
    }

    void closeReader(SegmentedByteBufReader reader) {
        this.readers.remove(reader);
    }

    public boolean isCompactable(long index) {
        long firstIndex = this.getCompactableIndex(index);
        return firstIndex != 0L && !this.segments.headMap((Object)firstIndex).isEmpty();
    }

    public long getCompactableIndex(long index) {
        Map.Entry segmentEntry = this.segments.floorEntry(index);
        return segmentEntry != null ? ((JournalSegment)segmentEntry.getValue()).firstIndex() : 0L;
    }

    public void compact(long index) {
        SortedMap compactSegments;
        long firstIndex = this.getCompactableIndex(index);
        if (firstIndex != 0L && !(compactSegments = this.segments.headMap((Object)firstIndex)).isEmpty()) {
            LOG.debug("{} - Compacting {} segment(s)", (Object)this.name, (Object)compactSegments.size());
            compactSegments.values().forEach(JournalSegment::delete);
            compactSegments.clear();
            this.resetHead(firstIndex);
        }
    }

    public void close() {
        if (this.currentSegment != null) {
            this.currentSegment = null;
            this.segments.values().forEach(JournalSegment::close);
            this.segments.clear();
        }
    }

    boolean isFlushOnCommit() {
        return this.flushOnCommit;
    }

    void setCommitIndex(long index) {
        this.commitIndex = index;
    }

    long getCommitIndex() {
        return this.commitIndex;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static final class Builder {
        private static final boolean DEFAULT_FLUSH_ON_COMMIT = false;
        private static final String DEFAULT_NAME = "atomix";
        private static final String DEFAULT_DIRECTORY = System.getProperty("user.dir");
        private static final int DEFAULT_MAX_SEGMENT_SIZE = 0x2000000;
        private static final int DEFAULT_MAX_ENTRY_SIZE = 0x100000;
        private static final int DEFAULT_MAX_ENTRIES_PER_SEGMENT = 0x100000;
        private static final double DEFAULT_INDEX_DENSITY = 0.005;
        private String name = "atomix";
        private StorageLevel storageLevel = StorageLevel.DISK;
        private File directory = new File(DEFAULT_DIRECTORY);
        private int maxSegmentSize = 0x2000000;
        private int maxEntrySize = 0x100000;
        private int maxEntriesPerSegment = 0x100000;
        private double indexDensity = 0.005;
        private boolean flushOnCommit = false;
        private ByteBufAllocator byteBufAllocator = ByteBufAllocator.DEFAULT;

        private Builder() {
        }

        public Builder withName(String name) {
            this.name = Objects.requireNonNull(name, "name cannot be null");
            return this;
        }

        public Builder withStorageLevel(StorageLevel storageLevel) {
            this.storageLevel = Objects.requireNonNull(storageLevel, "storageLevel cannot be null");
            return this;
        }

        public Builder withDirectory(String directory) {
            return this.withDirectory(new File(Objects.requireNonNull(directory, "directory cannot be null")));
        }

        public Builder withDirectory(File directory) {
            this.directory = Objects.requireNonNull(directory, "directory cannot be null");
            return this;
        }

        public Builder withMaxSegmentSize(int maxSegmentSize) {
            Preconditions.checkArgument((maxSegmentSize > 64 ? 1 : 0) != 0, (Object)"maxSegmentSize must be greater than 64");
            this.maxSegmentSize = maxSegmentSize;
            return this;
        }

        public Builder withMaxEntrySize(int maxEntrySize) {
            Preconditions.checkArgument((maxEntrySize > 0 ? 1 : 0) != 0, (Object)"maxEntrySize must be positive");
            this.maxEntrySize = maxEntrySize;
            return this;
        }

        @Deprecated(forRemoval=true, since="9.0.3")
        public Builder withMaxEntriesPerSegment(int maxEntriesPerSegment) {
            Preconditions.checkArgument((maxEntriesPerSegment > 0 ? 1 : 0) != 0, (Object)"max entries per segment must be positive");
            Preconditions.checkArgument((maxEntriesPerSegment <= 0x100000 ? 1 : 0) != 0, (Object)"max entries per segment cannot be greater than 1048576");
            this.maxEntriesPerSegment = maxEntriesPerSegment;
            return this;
        }

        public Builder withIndexDensity(double indexDensity) {
            Preconditions.checkArgument((indexDensity > 0.0 && indexDensity < 1.0 ? 1 : 0) != 0, (Object)"index density must be between 0 and 1");
            this.indexDensity = indexDensity;
            return this;
        }

        public Builder withFlushOnCommit() {
            return this.withFlushOnCommit(true);
        }

        public Builder withFlushOnCommit(boolean flushOnCommit) {
            this.flushOnCommit = flushOnCommit;
            return this;
        }

        public Builder withByteBufAllocator(ByteBufAllocator byteBufAllocator) {
            this.byteBufAllocator = Objects.requireNonNull(byteBufAllocator);
            return this;
        }

        public SegmentedByteBufJournal build() {
            return new SegmentedByteBufJournal(this.name, this.storageLevel, this.directory, this.maxSegmentSize, this.maxEntrySize, this.maxEntriesPerSegment, this.indexDensity, this.flushOnCommit, this.byteBufAllocator);
        }
    }
}

