/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.controller.cluster.access.client;

import akka.actor.ActorRef;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import org.checkerframework.checker.lock.qual.GuardedBy;
import org.checkerframework.checker.lock.qual.Holding;
import org.eclipse.jdt.annotation.NonNull;
import org.opendaylight.controller.cluster.access.client.AbstractReceivingClientConnection;
import org.opendaylight.controller.cluster.access.client.BackendInfo;
import org.opendaylight.controller.cluster.access.client.ClientActorBehavior;
import org.opendaylight.controller.cluster.access.client.ClientActorContext;
import org.opendaylight.controller.cluster.access.client.ConnectingClientConnection;
import org.opendaylight.controller.cluster.access.client.ConnectionEntry;
import org.opendaylight.controller.cluster.access.client.NoProgressException;
import org.opendaylight.controller.cluster.access.client.ReconnectForwarder;
import org.opendaylight.controller.cluster.access.client.RequestTimeoutException;
import org.opendaylight.controller.cluster.access.client.TransmitQueue;
import org.opendaylight.controller.cluster.access.client.TransmittedConnectionEntry;
import org.opendaylight.controller.cluster.access.concepts.Request;
import org.opendaylight.controller.cluster.access.concepts.RequestException;
import org.opendaylight.controller.cluster.access.concepts.Response;
import org.opendaylight.controller.cluster.access.concepts.ResponseEnvelope;
import org.opendaylight.controller.cluster.access.concepts.RuntimeRequestException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import scala.concurrent.duration.FiniteDuration;

public abstract sealed class AbstractClientConnection<T extends BackendInfo>
permits AbstractReceivingClientConnection, ConnectingClientConnection {
    private static final Logger LOG = LoggerFactory.getLogger(AbstractClientConnection.class);
    public static final long DEFAULT_BACKEND_ALIVE_TIMEOUT_NANOS = TimeUnit.SECONDS.toNanos(30L);
    public static final long DEFAULT_REQUEST_TIMEOUT_NANOS = TimeUnit.MINUTES.toNanos(2L);
    public static final long DEFAULT_NO_PROGRESS_TIMEOUT_NANOS = TimeUnit.MINUTES.toNanos(15L);
    private static final long DEBUG_DELAY_NANOS = TimeUnit.MILLISECONDS.toNanos(100L);
    private static final long MAX_DELAY_SECONDS = 5L;
    private static final long MAX_DELAY_NANOS = TimeUnit.SECONDS.toNanos(5L);
    private final Lock lock = new ReentrantLock();
    private final @NonNull ClientActorContext context;
    private final @NonNull Long cookie;
    private final String backendName;
    private final @GuardedBy(value={"lock"}) TransmitQueue queue;
    private @GuardedBy(value={"lock"}) boolean haveTimer;
    private long lastReceivedTicks;
    private volatile RequestException poisoned;

    private AbstractClientConnection(AbstractClientConnection<T> oldConn, TransmitQueue newQueue, String backendName) {
        this.context = oldConn.context;
        this.cookie = oldConn.cookie;
        this.backendName = Objects.requireNonNull(backendName);
        this.queue = Objects.requireNonNull(newQueue);
        this.lastReceivedTicks = oldConn.lastReceivedTicks;
    }

    AbstractClientConnection(ClientActorContext context, Long cookie, String backendName, int queueDepth) {
        this.context = Objects.requireNonNull(context);
        this.cookie = Objects.requireNonNull(cookie);
        this.backendName = Objects.requireNonNull(backendName);
        this.queue = new TransmitQueue.Halted(queueDepth);
        this.lastReceivedTicks = this.currentTime();
    }

    AbstractClientConnection(AbstractClientConnection<T> oldConn) {
        this(oldConn, new TransmitQueue.Halted(oldConn.queue, oldConn.currentTime()), oldConn.backendName);
    }

    AbstractClientConnection(AbstractClientConnection<T> oldConn, T newBackend, int queueDepth) {
        this(oldConn, new TransmitQueue.Transmitting(oldConn.queue, queueDepth, (BackendInfo)newBackend, oldConn.currentTime(), Objects.requireNonNull(oldConn.context).messageSlicer()), ((BackendInfo)newBackend).getName());
    }

    public final @NonNull ClientActorContext context() {
        return this.context;
    }

    public final @NonNull Long cookie() {
        return this.cookie;
    }

    public final @NonNull ActorRef localActor() {
        return this.context.self();
    }

    public final long currentTime() {
        return this.context.ticker().read();
    }

    public final void sendRequest(Request<?, ?> request, Consumer<Response<?, ?>> callback) {
        long now = this.currentTime();
        this.sendEntry(new ConnectionEntry(request, callback, now), now);
    }

    public final void enqueueRequest(Request<?, ?> request, Consumer<Response<?, ?>> callback, long enqueuedTicks) {
        this.enqueueEntry(new ConnectionEntry(request, callback, enqueuedTicks), this.currentTime());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long enqueueOrForward(ConnectionEntry entry, long now) {
        this.lock.lock();
        try {
            this.commonEnqueue(entry, now);
            long l = this.queue.enqueueOrForward(entry, now);
            return l;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final void enqueueEntry(ConnectionEntry entry, long now) {
        this.lock.lock();
        try {
            this.commonEnqueue(entry, now);
            this.queue.enqueueOrReplay(entry, now);
        }
        finally {
            this.lock.unlock();
        }
    }

    @Holding(value={"lock"})
    private void commonEnqueue(ConnectionEntry entry, long now) {
        RequestException maybePoison = this.poisoned;
        if (maybePoison != null) {
            throw new IllegalStateException("Connection " + String.valueOf(this) + " has been poisoned", (Throwable)maybePoison);
        }
        if (this.queue.isEmpty()) {
            this.scheduleTimer(entry.getEnqueuedTicks() + this.context.config().getRequestTimeout() - now);
        }
    }

    final void cancelDebt() {
        this.queue.cancelDebt(this.currentTime());
    }

    public abstract Optional<T> getBackendInfo();

    final Collection<ConnectionEntry> startReplay() {
        this.lock.lock();
        return this.queue.drain();
    }

    @Holding(value={"lock"})
    final void finishReplay(ReconnectForwarder forwarder) {
        this.setForwarder(forwarder);
        this.lastReceivedTicks = this.currentTime();
        this.lock.unlock();
    }

    @Holding(value={"lock"})
    final void setForwarder(ReconnectForwarder forwarder) {
        this.queue.setForwarder(forwarder, this.currentTime());
    }

    @Holding(value={"lock"})
    abstract ClientActorBehavior<T> lockedReconnect(ClientActorBehavior<T> var1, RequestException var2);

    final void sendEntry(ConnectionEntry entry, long now) {
        long delay = this.enqueueOrForward(entry, now);
        try {
            if (delay >= DEBUG_DELAY_NANOS) {
                if (delay > MAX_DELAY_NANOS) {
                    LOG.info("Capping {} throttle delay from {} to {} seconds", new Object[]{this, TimeUnit.NANOSECONDS.toSeconds(delay), 5L, new Throwable()});
                    delay = MAX_DELAY_NANOS;
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("{}: Sleeping for {}ms on connection {}", new Object[]{this.context.persistenceId(), TimeUnit.NANOSECONDS.toMillis(delay), this});
                }
            }
            TimeUnit.NANOSECONDS.sleep(delay);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOG.debug("Interrupted after sleeping {}ns", (Object)(this.currentTime() - now), (Object)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    final ClientActorBehavior<T> reconnect(ClientActorBehavior<T> current, RequestException cause) {
        this.lock.lock();
        try {
            ClientActorBehavior<T> clientActorBehavior = this.lockedReconnect(current, cause);
            return clientActorBehavior;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Holding(value={"lock"})
    private void scheduleTimer(long delay) {
        if (this.haveTimer) {
            LOG.debug("{}: timer already scheduled on {}", (Object)this.context.persistenceId(), (Object)this);
            return;
        }
        if (this.queue.hasSuccessor()) {
            LOG.debug("{}: connection {} has a successor, not scheduling timer", (Object)this.context.persistenceId(), (Object)this);
            return;
        }
        long normalized = delay <= 0L ? 0L : Math.min(delay, this.context.config().getBackendAlivenessTimerInterval());
        FiniteDuration dur = FiniteDuration.fromNanos((long)normalized);
        LOG.debug("{}: connection {} scheduling timeout in {}", new Object[]{this.context.persistenceId(), this, dur});
        this.context.executeInActor(this::runTimer, dur);
        this.haveTimer = true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    final ClientActorBehavior<T> runTimer(ClientActorBehavior<T> current) {
        List<ConnectionEntry> poisonEntries;
        NoProgressException poisonCause;
        this.lock.lock();
        try {
            this.haveTimer = false;
            long now = this.currentTime();
            LOG.debug("{}: running timer on {}", (Object)this.context.persistenceId(), (Object)this);
            long ticksSinceProgress = this.queue.ticksStalling(now);
            if (ticksSinceProgress < this.context.config().getNoProgressTimeout()) {
                OptionalLong delay = this.lockedCheckTimeout(now);
                if (delay == null) {
                    LOG.debug("{}: connection {} timed out", (Object)this.context.persistenceId(), (Object)this);
                    ClientActorBehavior<T> clientActorBehavior = this.lockedReconnect(current, (RequestException)new RuntimeRequestException("Backend connection timed out", (Throwable)new TimeoutException()));
                    return clientActorBehavior;
                }
                if (delay.isPresent()) {
                    this.scheduleTimer(delay.orElseThrow());
                } else {
                    LOG.debug("{}: not scheduling timeout on {}", (Object)this.context.persistenceId(), (Object)this);
                }
                ClientActorBehavior<T> clientActorBehavior = current;
                return clientActorBehavior;
            }
            LOG.error("Queue {} has not seen progress in {} seconds, failing all requests", (Object)this, (Object)TimeUnit.NANOSECONDS.toSeconds(ticksSinceProgress));
            poisonCause = new NoProgressException(ticksSinceProgress);
            poisonEntries = this.lockedPoison(poisonCause);
            current.removeConnection(this);
        }
        finally {
            this.lock.unlock();
        }
        AbstractClientConnection.poison(poisonEntries, poisonCause);
        return current;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    final OptionalLong checkTimeout(long now) {
        this.lock.lock();
        try {
            OptionalLong optionalLong = this.lockedCheckTimeout(now);
            return optionalLong;
        }
        finally {
            this.lock.unlock();
        }
    }

    long backendSilentTicks(long now) {
        return now - this.lastReceivedTicks;
    }

    @SuppressFBWarnings(value={"NP_OPTIONAL_RETURN_NULL"}, justification="Returning null Optional is documented in the API contract.")
    private @GuardedBy(value={"lock"}) OptionalLong lockedCheckTimeout(long now) {
        if (this.queue.isEmpty()) {
            LOG.debug("{}: connection {} is empty", (Object)this.context.persistenceId(), (Object)this);
            return OptionalLong.empty();
        }
        long backendSilentTicks = this.backendSilentTicks(now);
        if (backendSilentTicks >= this.context.config().getBackendAlivenessTimerInterval()) {
            LOG.debug("{}: Connection {} has not seen activity from backend for {} nanoseconds, timing out", new Object[]{this.context.persistenceId(), this, backendSilentTicks});
            return null;
        }
        int tasksTimedOut = 0;
        ConnectionEntry head = this.queue.peek();
        while (head != null) {
            long requestTimeout;
            long beenOpen = now - head.getEnqueuedTicks();
            if (beenOpen < (requestTimeout = this.context.config().getRequestTimeout())) {
                return OptionalLong.of(requestTimeout - beenOpen);
            }
            ++tasksTimedOut;
            this.queue.remove(now);
            LOG.debug("{}: Connection {} timed out entry {}", new Object[]{this.context.persistenceId(), this, head});
            this.timeoutEntry(head, beenOpen);
            head = this.queue.peek();
        }
        LOG.debug("Connection {} timed out {} tasks", (Object)this, (Object)tasksTimedOut);
        if (tasksTimedOut != 0) {
            this.queue.tryTransmit(now);
        }
        return OptionalLong.empty();
    }

    private void timeoutEntry(ConnectionEntry entry, long beenOpen) {
        this.context.executeInActor(current -> {
            double time = (double)beenOpen * 1.0 / 1.0E9;
            entry.complete((Response<?, ?>)entry.getRequest().toRequestFailure((RequestException)new RequestTimeoutException(String.valueOf(entry.getRequest()) + " timed out after " + time + " seconds. The backend for " + this.backendName + " is not available.")));
            return current;
        });
    }

    final void poison(RequestException cause) {
        List<ConnectionEntry> entries;
        this.lock.lock();
        try {
            entries = this.lockedPoison(cause);
        }
        finally {
            this.lock.unlock();
        }
        AbstractClientConnection.poison(entries, cause);
    }

    private static void poison(Collection<? extends ConnectionEntry> entries, RequestException cause) {
        for (ConnectionEntry connectionEntry : entries) {
            Request<?, ?> request = connectionEntry.getRequest();
            LOG.trace("Poisoning request {}", request, (Object)cause);
            connectionEntry.complete((Response<?, ?>)request.toRequestFailure(cause));
        }
    }

    @Holding(value={"lock"})
    private List<ConnectionEntry> lockedPoison(RequestException cause) {
        this.poisoned = this.enrichPoison(cause);
        return this.queue.poison();
    }

    RequestException enrichPoison(RequestException ex) {
        return ex;
    }

    @VisibleForTesting
    final RequestException poisoned() {
        return this.poisoned;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void receiveResponse(ResponseEnvelope<?> envelope) {
        Optional<TransmittedConnectionEntry> maybeEntry;
        long now;
        this.lastReceivedTicks = now = this.currentTime();
        this.lock.lock();
        try {
            maybeEntry = this.queue.complete(envelope, now);
        }
        finally {
            this.lock.unlock();
        }
        if (maybeEntry.isPresent()) {
            TransmittedConnectionEntry entry = maybeEntry.orElseThrow();
            LOG.debug("Completing {} with {}", (Object)entry, envelope);
            entry.complete((Response)envelope.getMessage());
        }
    }

    public final String toString() {
        return this.addToStringAttributes(MoreObjects.toStringHelper((Object)this).omitNullValues()).toString();
    }

    MoreObjects.ToStringHelper addToStringAttributes(MoreObjects.ToStringHelper toStringHelper) {
        return toStringHelper.add("client", (Object)this.context.getIdentifier()).add("cookie", (Object)this.cookie).add("poisoned", (Object)this.poisoned);
    }
}

