/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.controller.cluster.raft;

import akka.persistence.RecoveryCompleted;
import akka.persistence.SnapshotOffer;
import com.google.common.base.Stopwatch;
import java.util.Collections;
import java.util.concurrent.TimeUnit;
import org.opendaylight.controller.cluster.PersistentDataProvider;
import org.opendaylight.controller.cluster.raft.RaftActorContext;
import org.opendaylight.controller.cluster.raft.RaftActorRecoveryCohort;
import org.opendaylight.controller.cluster.raft.RaftEntryMeta;
import org.opendaylight.controller.cluster.raft.ReplicatedLog;
import org.opendaylight.controller.cluster.raft.ReplicatedLogEntry;
import org.opendaylight.controller.cluster.raft.ReplicatedLogImpl;
import org.opendaylight.controller.cluster.raft.SnapshotManager;
import org.opendaylight.controller.cluster.raft.base.messages.ApplySnapshot;
import org.opendaylight.controller.cluster.raft.messages.Payload;
import org.opendaylight.controller.cluster.raft.messages.PersistentPayload;
import org.opendaylight.controller.cluster.raft.persisted.ApplyJournalEntries;
import org.opendaylight.controller.cluster.raft.persisted.DeleteEntries;
import org.opendaylight.controller.cluster.raft.persisted.EmptyState;
import org.opendaylight.controller.cluster.raft.persisted.MigratedSerializable;
import org.opendaylight.controller.cluster.raft.persisted.ServerConfigurationPayload;
import org.opendaylight.controller.cluster.raft.persisted.Snapshot;
import org.opendaylight.controller.cluster.raft.persisted.UpdateElectionTerm;
import org.slf4j.Logger;

class RaftActorRecoverySupport {
    private final RaftActorContext context;
    private final RaftActorRecoveryCohort cohort;
    private int currentRecoveryBatchCount;
    private boolean dataRecoveredWithPersistenceDisabled;
    private boolean anyDataRecovered;
    private boolean hasMigratedDataRecovered;
    private Stopwatch recoveryTimer;
    private Stopwatch recoverySnapshotTimer;
    private final Logger log;

    RaftActorRecoverySupport(RaftActorContext context, RaftActorRecoveryCohort cohort) {
        this.context = context;
        this.cohort = cohort;
        this.log = context.getLogger();
    }

    boolean handleRecoveryMessage(Object message, PersistentDataProvider persistentProvider) {
        this.log.trace("{}: handleRecoveryMessage: {}", (Object)this.context.getId(), message);
        boolean bl = this.anyDataRecovered = this.anyDataRecovered || !(message instanceof RecoveryCompleted);
        if (RaftActorRecoverySupport.isMigratedSerializable(message)) {
            this.hasMigratedDataRecovered = true;
        }
        boolean recoveryComplete = false;
        if (message instanceof UpdateElectionTerm) {
            UpdateElectionTerm updateElectionTerm = (UpdateElectionTerm)message;
            this.context.getTermInformation().update(updateElectionTerm.getCurrentTerm(), updateElectionTerm.getVotedFor());
        } else if (message instanceof SnapshotOffer) {
            SnapshotOffer snapshotOffer = (SnapshotOffer)message;
            this.onRecoveredSnapshot(snapshotOffer);
        } else if (message instanceof ReplicatedLogEntry) {
            ReplicatedLogEntry replicatedLogEntry = (ReplicatedLogEntry)message;
            this.onRecoveredJournalLogEntry(replicatedLogEntry);
        } else if (message instanceof ApplyJournalEntries) {
            ApplyJournalEntries applyJournalEntries = (ApplyJournalEntries)message;
            this.onRecoveredApplyLogEntries(applyJournalEntries.getToIndex());
        } else if (message instanceof DeleteEntries) {
            DeleteEntries deleteEntries = (DeleteEntries)message;
            this.onDeleteEntries(deleteEntries);
        } else if (message instanceof ServerConfigurationPayload) {
            ServerConfigurationPayload serverConfigurationPayload = (ServerConfigurationPayload)message;
            this.context.updatePeerIds(serverConfigurationPayload);
        } else if (message instanceof RecoveryCompleted) {
            recoveryComplete = true;
            this.onRecoveryCompletedMessage(persistentProvider);
        }
        return recoveryComplete;
    }

    private void possiblyRestoreFromSnapshot() {
        Snapshot restoreFromSnapshot = this.cohort.getRestoreFromSnapshot();
        if (restoreFromSnapshot == null) {
            return;
        }
        if (this.anyDataRecovered) {
            this.log.warn("{}: The provided restore snapshot was not applied because the persistence store is not empty", (Object)this.context.getId());
            return;
        }
        this.log.debug("{}: Restore snapshot: {}", (Object)this.context.getId(), (Object)restoreFromSnapshot);
        this.context.getSnapshotManager().apply(new ApplySnapshot(restoreFromSnapshot));
    }

    private ReplicatedLog replicatedLog() {
        return this.context.getReplicatedLog();
    }

    private void initRecoveryTimers() {
        if (this.recoveryTimer == null) {
            this.recoveryTimer = Stopwatch.createStarted();
        }
        if (this.recoverySnapshotTimer == null && this.context.getConfigParams().getRecoverySnapshotIntervalSeconds() > 0) {
            this.recoverySnapshotTimer = Stopwatch.createStarted();
        }
    }

    private void onRecoveredSnapshot(SnapshotOffer offer) {
        this.log.debug("{}: SnapshotOffer called.", (Object)this.context.getId());
        this.initRecoveryTimers();
        Snapshot snapshot = (Snapshot)offer.snapshot();
        for (ReplicatedLogEntry entry : snapshot.getUnAppliedEntries()) {
            if (!RaftActorRecoverySupport.isMigratedPayload(entry)) continue;
            this.hasMigratedDataRecovered = true;
        }
        if (!this.context.getPersistenceProvider().isRecoveryApplicable()) {
            snapshot = Snapshot.create(EmptyState.INSTANCE, Collections.emptyList(), -1L, -1L, -1L, -1L, snapshot.getElectionTerm(), snapshot.getElectionVotedFor(), snapshot.getServerConfiguration());
        }
        this.context.setReplicatedLog(ReplicatedLogImpl.newInstance(snapshot, this.context));
        this.context.setLastApplied(snapshot.getLastAppliedIndex());
        this.context.setCommitIndex(snapshot.getLastAppliedIndex());
        this.context.getTermInformation().update(snapshot.getElectionTerm(), snapshot.getElectionVotedFor());
        Stopwatch timer = Stopwatch.createStarted();
        Snapshot.State snapshotState = snapshot.getState();
        if (snapshotState.needsMigration()) {
            this.hasMigratedDataRecovered = true;
        }
        if (!(snapshotState instanceof EmptyState)) {
            this.cohort.applyRecoverySnapshot(snapshotState);
        }
        if (snapshot.getServerConfiguration() != null) {
            this.context.updatePeerIds(snapshot.getServerConfiguration());
        }
        timer.stop();
        this.log.info("Recovery snapshot applied for {} in {}: snapshotIndex={}, snapshotTerm={}, journal-size={}", new Object[]{this.context.getId(), timer, this.replicatedLog().getSnapshotIndex(), this.replicatedLog().getSnapshotTerm(), this.replicatedLog().size()});
    }

    private void onRecoveredJournalLogEntry(ReplicatedLogEntry logEntry) {
        Payload payload;
        if (this.log.isDebugEnabled()) {
            this.log.debug("{}: Received ReplicatedLogEntry for recovery: index: {}, size: {}", new Object[]{this.context.getId(), logEntry.index(), logEntry.size()});
        }
        if ((payload = logEntry.getData()) instanceof ServerConfigurationPayload) {
            ServerConfigurationPayload payload2 = (ServerConfigurationPayload)payload;
            this.context.updatePeerIds(payload2);
        }
        if (RaftActorRecoverySupport.isMigratedPayload(logEntry)) {
            this.hasMigratedDataRecovered = true;
        }
        if (this.context.getPersistenceProvider().isRecoveryApplicable()) {
            this.replicatedLog().append(logEntry);
        } else if (!RaftActorRecoverySupport.isPersistentPayload(logEntry)) {
            this.dataRecoveredWithPersistenceDisabled = true;
        }
    }

    private void onRecoveredApplyLogEntries(long toIndex) {
        if (!this.context.getPersistenceProvider().isRecoveryApplicable()) {
            this.dataRecoveredWithPersistenceDisabled = true;
            return;
        }
        long lastUnappliedIndex = this.context.getLastApplied() + 1L;
        if (this.log.isDebugEnabled()) {
            this.log.debug("{}: Received apply journal entries for recovery, applying to state: {} to {}", new Object[]{this.context.getId(), lastUnappliedIndex, toIndex});
        }
        long lastApplied = lastUnappliedIndex - 1L;
        for (long i = lastUnappliedIndex; i <= toIndex; ++i) {
            ReplicatedLogEntry logEntry = this.replicatedLog().get(i);
            if (logEntry != null) {
                ++lastApplied;
                this.batchRecoveredLogEntry(logEntry);
                if (!this.shouldTakeRecoverySnapshot() || this.context.getSnapshotManager().isCapturing()) continue;
                if (this.currentRecoveryBatchCount > 0) {
                    this.endCurrentLogRecoveryBatch();
                }
                this.context.setLastApplied(lastApplied);
                this.context.setCommitIndex(lastApplied);
                this.takeRecoverySnapshot(logEntry);
                continue;
            }
            this.log.error("{}: Log entry not found for index {}", (Object)this.context.getId(), (Object)i);
            break;
        }
        this.context.setLastApplied(lastApplied);
        this.context.setCommitIndex(lastApplied);
    }

    private void onDeleteEntries(DeleteEntries deleteEntries) {
        if (this.context.getPersistenceProvider().isRecoveryApplicable()) {
            this.replicatedLog().removeFrom(deleteEntries.getFromIndex());
        } else {
            this.dataRecoveredWithPersistenceDisabled = true;
        }
    }

    private void batchRecoveredLogEntry(ReplicatedLogEntry logEntry) {
        this.initRecoveryTimers();
        if (logEntry.getData() instanceof ServerConfigurationPayload) {
            return;
        }
        int batchSize = this.context.getConfigParams().getJournalRecoveryLogBatchSize();
        if (this.currentRecoveryBatchCount == 0) {
            this.cohort.startLogRecoveryBatch(batchSize);
        }
        this.cohort.appendRecoveredLogEntry(logEntry.getData());
        if (++this.currentRecoveryBatchCount >= batchSize) {
            this.endCurrentLogRecoveryBatch();
        }
    }

    private void takeRecoverySnapshot(RaftEntryMeta logEntry) {
        this.log.info("Time for recovery snapshot on entry with index {}", (Object)logEntry.index());
        SnapshotManager snapshotManager = this.context.getSnapshotManager();
        if (snapshotManager.capture(logEntry, -1L)) {
            this.log.info("Capturing snapshot, resetting timer for the next recovery snapshot interval.");
            this.recoverySnapshotTimer.reset().start();
        } else {
            this.log.info("SnapshotManager is not able to capture snapshot at this time. It will be retried again with the next recovered entry.");
        }
    }

    private boolean shouldTakeRecoverySnapshot() {
        return this.recoverySnapshotTimer != null && this.recoverySnapshotTimer.elapsed(TimeUnit.SECONDS) >= (long)this.context.getConfigParams().getRecoverySnapshotIntervalSeconds();
    }

    private void endCurrentLogRecoveryBatch() {
        this.cohort.applyCurrentLogRecoveryBatch();
        this.currentRecoveryBatchCount = 0;
    }

    private void onRecoveryCompletedMessage(PersistentDataProvider persistentProvider) {
        Object recoveryTime;
        if (this.currentRecoveryBatchCount > 0) {
            this.endCurrentLogRecoveryBatch();
        }
        if (this.recoveryTimer != null) {
            recoveryTime = " in " + String.valueOf(this.recoveryTimer.stop());
            this.recoveryTimer = null;
        } else {
            recoveryTime = "";
        }
        if (this.recoverySnapshotTimer != null) {
            this.recoverySnapshotTimer.stop();
            this.recoverySnapshotTimer = null;
        }
        this.log.info("{}: Recovery completed {} - Switching actor to Follower - last log index = {}, last log term = {}, snapshot index = {}, snapshot term = {}, journal size = {}", new Object[]{this.context.getId(), recoveryTime, this.replicatedLog().lastIndex(), this.replicatedLog().lastTerm(), this.replicatedLog().getSnapshotIndex(), this.replicatedLog().getSnapshotTerm(), this.replicatedLog().size()});
        if (this.dataRecoveredWithPersistenceDisabled || this.hasMigratedDataRecovered && !this.context.getPersistenceProvider().isRecoveryApplicable()) {
            if (this.hasMigratedDataRecovered) {
                this.log.info("{}: Saving snapshot after recovery due to migrated messages", (Object)this.context.getId());
            } else {
                this.log.info("{}: Saving snapshot after recovery due to data persistence disabled", (Object)this.context.getId());
            }
            Snapshot snapshot = Snapshot.create(EmptyState.INSTANCE, Collections.emptyList(), -1L, -1L, -1L, -1L, this.context.getTermInformation().getCurrentTerm(), this.context.getTermInformation().getVotedFor(), this.context.getPeerServerInfo(true));
            persistentProvider.saveSnapshot((Object)snapshot);
            persistentProvider.deleteMessages(persistentProvider.getLastSequenceNumber());
        } else if (this.hasMigratedDataRecovered) {
            this.log.info("{}: Snapshot capture initiated after recovery due to migrated messages", (Object)this.context.getId());
            this.context.getSnapshotManager().capture(this.replicatedLog().lastMeta(), -1L);
        } else {
            this.possiblyRestoreFromSnapshot();
        }
    }

    private static boolean isPersistentPayload(ReplicatedLogEntry repLogEntry) {
        return repLogEntry.getData() instanceof PersistentPayload;
    }

    private static boolean isMigratedPayload(ReplicatedLogEntry repLogEntry) {
        return RaftActorRecoverySupport.isMigratedSerializable(repLogEntry.getData());
    }

    private static boolean isMigratedSerializable(Object message) {
        MigratedSerializable migrated;
        return message instanceof MigratedSerializable && (migrated = (MigratedSerializable)message).isMigrated();
    }
}

