/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.controller.cluster.common.actor;

import com.google.common.annotations.Beta;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.base.Ticker;
import com.google.common.collect.ImmutableList;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Beta
public final class MessageTracker {
    private static final Logger LOG = LoggerFactory.getLogger(MessageTracker.class);
    private static final Context NO_OP_CONTEXT = new Context(){

        @Override
        public void close() {
        }

        @Override
        public Optional<Error> error() {
            return Optional.empty();
        }
    };
    private final List<MessageProcessingTime> messagesSinceLastExpectedMessage = new LinkedList<MessageProcessingTime>();
    private final CurrentMessageContext currentMessageContext;
    private final Stopwatch expectedMessageWatch;
    private final Class<?> expectedMessageClass;
    private final long expectedArrivalInterval;
    private final Ticker ticker;
    private Object lastExpectedMessage;

    @VisibleForTesting
    MessageTracker(Class<?> expectedMessageClass, long expectedArrivalIntervalInMillis, Ticker ticker) {
        Preconditions.checkArgument((expectedArrivalIntervalInMillis >= 0L ? 1 : 0) != 0);
        this.expectedMessageClass = Objects.requireNonNull(expectedMessageClass);
        this.expectedArrivalInterval = TimeUnit.MILLISECONDS.toNanos(expectedArrivalIntervalInMillis);
        this.ticker = Objects.requireNonNull(ticker);
        this.expectedMessageWatch = Stopwatch.createUnstarted((Ticker)ticker);
        this.currentMessageContext = new CurrentMessageContext();
    }

    public MessageTracker(Class<?> expectedMessageClass, long expectedArrivalIntervalInMillis) {
        this(expectedMessageClass, expectedArrivalIntervalInMillis, Ticker.systemTicker());
    }

    public void begin() {
        if (!this.expectedMessageWatch.isRunning()) {
            LOG.trace("Started tracking class {} timeout {}ns", this.expectedMessageClass, (Object)this.expectedArrivalInterval);
            this.expectedMessageWatch.start();
        }
    }

    public Context received(Object message) {
        if (!this.expectedMessageWatch.isRunning()) {
            return NO_OP_CONTEXT;
        }
        if (this.expectedMessageClass.isInstance(message)) {
            long actualElapsedTime = this.expectedMessageWatch.elapsed(TimeUnit.NANOSECONDS);
            if (actualElapsedTime > this.expectedArrivalInterval) {
                return new ErrorContext(message, new FailedExpectation(this.lastExpectedMessage, message, this.messagesSinceLastExpectedMessage, this.expectedArrivalInterval, actualElapsedTime));
            }
            this.lastExpectedMessage = message;
            this.messagesSinceLastExpectedMessage.clear();
            this.expectedMessageWatch.reset().start();
        }
        this.currentMessageContext.reset(message);
        return this.currentMessageContext;
    }

    void processed(Object message, long messageElapseTimeInNanos) {
        if (this.expectedMessageWatch.isRunning() && !this.expectedMessageClass.isInstance(message)) {
            this.messagesSinceLastExpectedMessage.add(new MessageProcessingTime(message.getClass(), messageElapseTimeInNanos));
        }
    }

    public List<MessageProcessingTime> getMessagesSinceLastExpectedMessage() {
        return ImmutableList.copyOf(this.messagesSinceLastExpectedMessage);
    }

    private final class CurrentMessageContext
    extends AbstractTimedContext {
        private final Stopwatch stopwatch;
        private Object message;

        private CurrentMessageContext() {
            this.stopwatch = Stopwatch.createUnstarted((Ticker)MessageTracker.this.ticker);
        }

        void reset(Object newMessage) {
            this.message = Objects.requireNonNull(newMessage);
            Preconditions.checkState((!this.stopwatch.isRunning() ? 1 : 0) != 0, (String)"Trying to reset a context that is not done (%s). currentMessage = %s", (Object)this, (Object)newMessage);
            this.stopwatch.start();
        }

        @Override
        Object message() {
            return this.message;
        }

        @Override
        Stopwatch stopTimer() {
            return this.stopwatch.stop();
        }

        @Override
        public Optional<Error> error() {
            return Optional.empty();
        }
    }

    public static abstract class Context
    implements AutoCloseable {
        Context() {
        }

        public abstract Optional<Error> error();

        @Override
        public abstract void close();
    }

    private final class ErrorContext
    extends AbstractTimedContext {
        private final Stopwatch stopwatch;
        private final Object message;
        private final Error error;

        ErrorContext(Object message, Error error) {
            this.stopwatch = Stopwatch.createStarted((Ticker)MessageTracker.this.ticker);
            this.message = Objects.requireNonNull(message);
            this.error = Objects.requireNonNull(error);
        }

        @Override
        Object message() {
            return this.message;
        }

        @Override
        Stopwatch stopTimer() {
            return this.stopwatch.stop();
        }

        @Override
        public Optional<Error> error() {
            return Optional.of(this.error);
        }
    }

    private static final class FailedExpectation
    implements Error {
        private final Object lastExpectedMessage;
        private final Object currentExpectedMessage;
        private final List<MessageProcessingTime> messagesSinceLastExpectedMessage;
        private final long expectedTimeInMillis;
        private final long actualTimeInMillis;

        FailedExpectation(Object lastExpectedMessage, Object message, List<MessageProcessingTime> messagesSinceLastExpectedMessage, long expectedTimeNanos, long actualTimeNanos) {
            this.lastExpectedMessage = lastExpectedMessage;
            this.currentExpectedMessage = message;
            this.messagesSinceLastExpectedMessage = ImmutableList.copyOf(messagesSinceLastExpectedMessage);
            this.expectedTimeInMillis = TimeUnit.NANOSECONDS.toMillis(expectedTimeNanos);
            this.actualTimeInMillis = TimeUnit.NANOSECONDS.toMillis(actualTimeNanos);
        }

        @Override
        public Object getLastExpectedMessage() {
            return this.lastExpectedMessage;
        }

        @Override
        public Object getCurrentExpectedMessage() {
            return this.currentExpectedMessage;
        }

        @Override
        public List<MessageProcessingTime> getMessageProcessingTimesSinceLastExpectedMessage() {
            return this.messagesSinceLastExpectedMessage;
        }

        public String toString() {
            StringBuilder builder = new StringBuilder().append("\n> Last Expected Message = ").append(this.lastExpectedMessage).append("\n> Current Expected Message = ").append(this.currentExpectedMessage).append("\n> Expected time in between messages = ").append(this.expectedTimeInMillis).append("\n> Actual time in between messages = ").append(this.actualTimeInMillis);
            for (MessageProcessingTime time : this.messagesSinceLastExpectedMessage) {
                builder.append("\n\t> ").append(time);
            }
            return builder.toString();
        }
    }

    public static interface Error {
        public Object getLastExpectedMessage();

        public Object getCurrentExpectedMessage();

        public List<MessageProcessingTime> getMessageProcessingTimesSinceLastExpectedMessage();
    }

    public static final class MessageProcessingTime {
        private final Class<?> messageClass;
        private final long elapsedTimeInNanos;

        MessageProcessingTime(Class<?> messageClass, long elapsedTimeInNanos) {
            this.messageClass = Objects.requireNonNull(messageClass);
            this.elapsedTimeInNanos = elapsedTimeInNanos;
        }

        public String toString() {
            return "MessageProcessingTime [messageClass=" + String.valueOf(this.messageClass) + ", elapsedTimeInMillis=" + TimeUnit.NANOSECONDS.toMillis(this.elapsedTimeInNanos) + "]";
        }

        public Class<?> getMessageClass() {
            return this.messageClass;
        }

        public long getElapsedTimeInNanos() {
            return this.elapsedTimeInNanos;
        }
    }

    private abstract class AbstractTimedContext
    extends Context {
        private AbstractTimedContext() {
        }

        abstract Object message();

        abstract Stopwatch stopTimer();

        @Override
        public final void close() {
            MessageTracker.this.processed(this.message(), this.stopTimer().elapsed(TimeUnit.NANOSECONDS));
        }
    }
}

