/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.controller.remote.rpc.registry.gossip;

import akka.actor.ActorRef;
import akka.actor.ActorRefProvider;
import akka.actor.ActorSelection;
import akka.actor.ActorSystem;
import akka.actor.Address;
import akka.actor.Cancellable;
import akka.actor.Props;
import akka.cluster.Cluster;
import akka.cluster.ClusterActorRefProvider;
import akka.cluster.ClusterEvent;
import akka.cluster.Member;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Verify;
import com.google.common.collect.Maps;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import org.opendaylight.controller.cluster.common.actor.AbstractUntypedActorWithMetering;
import org.opendaylight.controller.remote.rpc.RemoteOpsProviderConfig;
import org.opendaylight.controller.remote.rpc.registry.gossip.Bucket;
import org.opendaylight.controller.remote.rpc.registry.gossip.BucketStoreAccess;
import org.opendaylight.controller.remote.rpc.registry.gossip.GossipEnvelope;
import org.opendaylight.controller.remote.rpc.registry.gossip.GossipStatus;
import scala.concurrent.ExecutionContext;
import scala.concurrent.duration.FiniteDuration;

public class Gossiper
extends AbstractUntypedActorWithMetering {
    private static final Object GOSSIP_TICK = new Object(){

        public String toString() {
            return "gossip tick";
        }
    };
    private final boolean autoStartGossipTicks;
    private final RemoteOpsProviderConfig config;
    private final List<Address> clusterMembers = new ArrayList<Address>();
    private final Map<Address, ActorSelection> peers = new HashMap<Address, ActorSelection>();
    private Address selfAddress;
    private Cluster cluster;
    private Cancellable gossipTask;
    private BucketStoreAccess bucketStore;

    Gossiper(RemoteOpsProviderConfig config, Boolean autoStartGossipTicks) {
        this.config = Objects.requireNonNull(config);
        this.autoStartGossipTicks = autoStartGossipTicks;
    }

    Gossiper(RemoteOpsProviderConfig config) {
        this(config, Boolean.TRUE);
    }

    public static Props props(RemoteOpsProviderConfig config) {
        return Props.create(Gossiper.class, (Object[])new Object[]{config});
    }

    static Props testProps(RemoteOpsProviderConfig config) {
        return Props.create(Gossiper.class, (Object[])new Object[]{config, Boolean.FALSE});
    }

    public void preStart() {
        ActorRefProvider provider = this.getContext().provider();
        this.selfAddress = provider.getDefaultAddress();
        this.bucketStore = new BucketStoreAccess(this.getContext().parent(), (ExecutionContext)this.getContext().dispatcher(), this.config.getAskDuration());
        if (provider instanceof ClusterActorRefProvider) {
            this.cluster = Cluster.get((ActorSystem)this.getContext().system());
            this.cluster.subscribe(this.getSelf(), (ClusterEvent.SubscriptionInitialStateMode)ClusterEvent.initialStateAsEvents(), new Class[]{ClusterEvent.MemberEvent.class, ClusterEvent.ReachableMember.class, ClusterEvent.UnreachableMember.class});
        }
        if (this.autoStartGossipTicks) {
            this.gossipTask = this.getContext().system().scheduler().scheduleAtFixedRate(new FiniteDuration(1L, TimeUnit.SECONDS), this.config.getGossipTickInterval(), this.getSelf(), GOSSIP_TICK, (ExecutionContext)this.getContext().dispatcher(), this.getSelf());
        }
    }

    public void postStop() {
        if (this.cluster != null) {
            this.cluster.unsubscribe(this.getSelf());
        }
        if (this.gossipTask != null) {
            this.gossipTask.cancel();
        }
    }

    protected void handleReceive(Object message) {
        if (GOSSIP_TICK.equals(message)) {
            this.receiveGossipTick();
        } else if (message instanceof GossipStatus) {
            GossipStatus status = (GossipStatus)message;
            this.receiveGossipStatus(status);
        } else if (message instanceof GossipEnvelope) {
            GossipEnvelope envelope = (GossipEnvelope)message;
            this.receiveGossip(envelope);
        } else if (message instanceof ClusterEvent.MemberUp) {
            ClusterEvent.MemberUp memberUp = (ClusterEvent.MemberUp)message;
            this.receiveMemberUpOrReachable(memberUp.member());
        } else if (message instanceof ClusterEvent.ReachableMember) {
            ClusterEvent.ReachableMember reachableMember = (ClusterEvent.ReachableMember)message;
            this.receiveMemberUpOrReachable(reachableMember.member());
        } else if (message instanceof ClusterEvent.MemberRemoved) {
            ClusterEvent.MemberRemoved memberRemoved = (ClusterEvent.MemberRemoved)message;
            this.receiveMemberRemoveOrUnreachable(memberRemoved.member());
        } else if (message instanceof ClusterEvent.UnreachableMember) {
            ClusterEvent.UnreachableMember unreachableMember = (ClusterEvent.UnreachableMember)message;
            this.receiveMemberRemoveOrUnreachable(unreachableMember.member());
        } else {
            this.unhandled(message);
        }
    }

    private void receiveMemberRemoveOrUnreachable(Member member) {
        this.LOG.debug("Received memberDown or Unreachable: {}", (Object)member);
        if (this.selfAddress.equals((Object)member.address())) {
            this.getContext().stop(this.getSelf());
            return;
        }
        this.removePeer(member.address());
        this.LOG.debug("Removed member [{}], Active member list [{}]", (Object)member.address(), this.clusterMembers);
    }

    private void addPeer(Address address) {
        if (!this.clusterMembers.contains(address)) {
            this.clusterMembers.add(address);
        }
        this.peers.computeIfAbsent(address, input -> this.getContext().system().actorSelection(input.toString() + this.getSelf().path().toStringWithoutAddress()));
    }

    private void removePeer(Address address) {
        this.clusterMembers.remove(address);
        this.peers.remove(address);
        this.bucketStore.removeRemoteBucket(address);
    }

    private void receiveMemberUpOrReachable(Member member) {
        this.LOG.debug("Received memberUp or reachable: {}", (Object)member);
        if (this.selfAddress.equals((Object)member.address())) {
            return;
        }
        this.addPeer(member.address());
        this.LOG.debug("Added member [{}], Active member list [{}]", (Object)member.address(), this.clusterMembers);
    }

    @VisibleForTesting
    void receiveGossipTick() {
        Address address;
        switch (this.clusterMembers.size()) {
            case 0: {
                return;
            }
            case 1: {
                address = this.clusterMembers.get(0);
                break;
            }
            default: {
                int randomIndex = ThreadLocalRandom.current().nextInt(0, this.clusterMembers.size());
                address = this.clusterMembers.get(randomIndex);
            }
        }
        this.LOG.trace("Gossiping to [{}]", (Object)address);
        this.getLocalStatusAndSendTo((ActorSelection)Verify.verifyNotNull((Object)this.peers.get(address)));
    }

    @VisibleForTesting
    void receiveGossipStatus(GossipStatus status) {
        if (this.peers.containsKey(status.from())) {
            ActorRef sender = this.getSender();
            this.bucketStore.getBucketVersions(versions -> this.processRemoteStatus(sender, status, (Map<Address, Long>)versions));
        }
    }

    private void processRemoteStatus(ActorRef remote, GossipStatus status, Map<Address, Long> localVersions) {
        Map<Address, Long> remoteVersions = status.versions();
        HashSet<Address> localIsOlder = new HashSet<Address>(remoteVersions.keySet());
        localIsOlder.removeAll(localVersions.keySet());
        HashSet<Address> localIsNewer = new HashSet<Address>(localVersions.keySet());
        localIsNewer.removeAll(remoteVersions.keySet());
        for (Map.Entry<Address, Long> entry : remoteVersions.entrySet()) {
            Address address = entry.getKey();
            Long remoteVersion = entry.getValue();
            Long localVersion = localVersions.get(address);
            if (localVersion == null || remoteVersion == null) continue;
            if (localVersion < remoteVersion) {
                localIsOlder.add(address);
                continue;
            }
            if (localVersion <= remoteVersion) continue;
            localIsNewer.add(address);
        }
        if (!localIsOlder.isEmpty()) {
            remote.tell((Object)new GossipStatus(this.selfAddress, localVersions), this.getSelf());
        }
        if (!localIsNewer.isEmpty()) {
            this.bucketStore.getBucketsByMembers(localIsNewer, buckets -> {
                this.LOG.trace("Buckets to send from {}: {}", (Object)this.selfAddress, buckets);
                remote.tell((Object)new GossipEnvelope(this.selfAddress, remote.path().address(), (Map<Address, ? extends Bucket<?>>)buckets), this.getSelf());
            });
        }
    }

    @VisibleForTesting
    void receiveGossip(GossipEnvelope envelope) {
        if (!this.selfAddress.equals((Object)envelope.to())) {
            this.LOG.trace("Ignoring message intended for someone else. From [{}] to [{}]", (Object)envelope.from(), (Object)envelope.to());
            return;
        }
        this.updateRemoteBuckets(envelope.buckets());
    }

    @VisibleForTesting
    void updateRemoteBuckets(Map<Address, ? extends Bucket<?>> buckets) {
        this.bucketStore.updateRemoteBuckets(Maps.filterKeys(buckets, this.peers::containsKey));
    }

    @VisibleForTesting
    void getLocalStatusAndSendTo(ActorSelection remoteGossiper) {
        this.bucketStore.getBucketVersions(versions -> {
            this.LOG.trace("Sending bucket versions to [{}]", (Object)remoteGossiper);
            remoteGossiper.tell((Object)new GossipStatus(this.selfAddress, (Map<Address, Long>)versions), this.getSelf());
        });
    }

    @VisibleForTesting
    void setClusterMembers(Address ... members) {
        this.clusterMembers.clear();
        this.peers.clear();
        for (Address addr : members) {
            this.addPeer(addr);
        }
    }
}

