/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.controller.cluster.datastore;

import akka.actor.ActorSelection;
import akka.dispatch.Futures;
import akka.dispatch.OnComplete;
import com.google.common.base.Preconditions;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicReferenceFieldUpdater;
import org.opendaylight.controller.cluster.access.concepts.LocalHistoryIdentifier;
import org.opendaylight.controller.cluster.access.concepts.TransactionIdentifier;
import org.opendaylight.controller.cluster.datastore.AbstractTransactionContextFactory;
import org.opendaylight.controller.cluster.datastore.LocalTransactionChain;
import org.opendaylight.controller.cluster.datastore.TransactionContextFactory;
import org.opendaylight.controller.cluster.datastore.TransactionProxy;
import org.opendaylight.controller.cluster.datastore.TransactionType;
import org.opendaylight.controller.cluster.datastore.messages.CloseTransactionChain;
import org.opendaylight.controller.cluster.datastore.messages.PrimaryShardInfo;
import org.opendaylight.mdsal.dom.api.DOMTransactionChainClosedException;
import org.opendaylight.mdsal.dom.spi.store.DOMStoreReadTransaction;
import org.opendaylight.mdsal.dom.spi.store.DOMStoreReadWriteTransaction;
import org.opendaylight.mdsal.dom.spi.store.DOMStoreTransactionChain;
import org.opendaylight.mdsal.dom.spi.store.DOMStoreWriteTransaction;
import org.opendaylight.yangtools.yang.data.tree.api.ReadOnlyDataTree;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import scala.Function1;
import scala.concurrent.ExecutionContext;
import scala.concurrent.Future;
import scala.concurrent.Promise;

final class TransactionChainProxy
extends AbstractTransactionContextFactory<LocalTransactionChain>
implements DOMStoreTransactionChain {
    private static final State IDLE_STATE = new DefaultState(){

        @Override
        void checkReady() {
        }
    };
    private static final State CLOSED_STATE = new DefaultState(){

        @Override
        void checkReady() {
            throw new DOMTransactionChainClosedException("Transaction chain has been closed");
        }
    };
    private static final Logger LOG = LoggerFactory.getLogger(TransactionChainProxy.class);
    private static final AtomicReferenceFieldUpdater<TransactionChainProxy, State> STATE_UPDATER = AtomicReferenceFieldUpdater.newUpdater(TransactionChainProxy.class, State.class, "currentState");
    private final TransactionContextFactory parent;
    private volatile State currentState = IDLE_STATE;
    private final ConcurrentMap<TransactionIdentifier, Promise<Object>> priorReadOnlyTxPromises = new ConcurrentHashMap<TransactionIdentifier, Promise<Object>>();

    TransactionChainProxy(TransactionContextFactory parent, LocalHistoryIdentifier historyId) {
        super(parent.getActorUtils(), historyId);
        this.parent = parent;
    }

    public DOMStoreReadTransaction newReadOnlyTransaction() {
        this.currentState.checkReady();
        TransactionProxy transactionProxy = new TransactionProxy(this, TransactionType.READ_ONLY);
        this.priorReadOnlyTxPromises.put((TransactionIdentifier)transactionProxy.getIdentifier(), (Promise<Object>)Futures.promise());
        return transactionProxy;
    }

    public DOMStoreReadWriteTransaction newReadWriteTransaction() {
        this.getActorUtils().acquireTxCreationPermit();
        return this.allocateWriteTransaction(TransactionType.READ_WRITE);
    }

    public DOMStoreWriteTransaction newWriteOnlyTransaction() {
        this.getActorUtils().acquireTxCreationPermit();
        return this.allocateWriteTransaction(TransactionType.WRITE_ONLY);
    }

    @Override
    public void close() {
        this.currentState = CLOSED_STATE;
        this.getActorUtils().broadcast(version -> new CloseTransactionChain(this.getHistoryId(), (short)version).toSerializable(), CloseTransactionChain.class);
    }

    private TransactionProxy allocateWriteTransaction(TransactionType type) {
        State localState = this.currentState;
        localState.checkReady();
        TransactionProxy ret = new TransactionProxy(this, type);
        this.currentState = new Allocated((TransactionIdentifier)ret.getIdentifier(), localState.previousFuture());
        return ret;
    }

    @Override
    protected LocalTransactionChain factoryForShard(String shardName, ActorSelection shardLeader, ReadOnlyDataTree dataTree) {
        LocalTransactionChain ret = new LocalTransactionChain(this, shardLeader, dataTree);
        LOG.debug("Allocated transaction chain {} for shard {} leader {}", new Object[]{ret, shardName, shardLeader});
        return ret;
    }

    @Override
    protected Future<PrimaryShardInfo> findPrimaryShard(final String shardName, final TransactionIdentifier txId) {
        String previousTransactionId;
        State localState = this.currentState;
        Future<?> previous = localState.previousFuture();
        if (previous == null) {
            return this.combineFutureWithPossiblePriorReadOnlyTxFutures(this.parent.findPrimaryShard(shardName, txId), txId);
        }
        if (localState instanceof Pending) {
            previousTransactionId = ((Pending)localState).getIdentifier().toString();
            LOG.debug("Tx: {} - waiting for ready futures with pending Tx {}", (Object)txId, (Object)previousTransactionId);
        } else {
            previousTransactionId = "";
            LOG.debug("Waiting for ready futures on chain {}", (Object)this.getHistoryId());
        }
        previous = this.combineFutureWithPossiblePriorReadOnlyTxFutures(previous, txId);
        final Promise returnPromise = Futures.promise();
        OnComplete onComplete = new OnComplete(){

            public void onComplete(Throwable failure, Object notUsed) {
                if (failure != null) {
                    LOG.error("Tx: {} - ready future failed for previous Tx {}", (Object)txId, (Object)previousTransactionId);
                    returnPromise.failure(failure);
                } else {
                    LOG.debug("Tx: {} - previous Tx {} readied - proceeding to FindPrimaryShard", (Object)txId, (Object)previousTransactionId);
                    returnPromise.completeWith(TransactionChainProxy.this.parent.findPrimaryShard(shardName, txId));
                }
            }
        };
        previous.onComplete((Function1)onComplete, this.getActorUtils().getClientDispatcher());
        return returnPromise.future();
    }

    private <T> Future<T> combineFutureWithPossiblePriorReadOnlyTxFutures(Future<T> future, TransactionIdentifier txId) {
        return this.priorReadOnlyTxPromises.isEmpty() || this.priorReadOnlyTxPromises.containsKey(txId) ? future : this.combineWithPriorReadOnlyTxFutures(future, txId);
    }

    private <T> Future<T> combineWithPriorReadOnlyTxFutures(final Future<T> future, final TransactionIdentifier txId) {
        ArrayList priorReadOnlyTxPromiseEntries = new ArrayList(this.priorReadOnlyTxPromises.entrySet());
        if (priorReadOnlyTxPromiseEntries.isEmpty()) {
            return future;
        }
        ArrayList<Future> priorReadOnlyTxFutures = new ArrayList<Future>(priorReadOnlyTxPromiseEntries.size());
        for (Map.Entry entry : priorReadOnlyTxPromiseEntries) {
            LOG.debug("Tx: {} - waiting on future for prior read-only Tx {}", (Object)txId, entry.getKey());
            priorReadOnlyTxFutures.add(((Promise)entry.getValue()).future());
        }
        Future combinedFutures = Futures.sequence(priorReadOnlyTxFutures, (ExecutionContext)this.getActorUtils().getClientDispatcher());
        final Promise promise = Futures.promise();
        OnComplete<Iterable<Object>> onComplete = new OnComplete<Iterable<Object>>(){

            public void onComplete(Throwable failure, Iterable<Object> notUsed) {
                LOG.debug("Tx: {} - prior read-only Tx futures complete", (Object)txId);
                promise.completeWith(future);
            }
        };
        combinedFutures.onComplete((Function1)onComplete, this.getActorUtils().getClientDispatcher());
        return promise.future();
    }

    @Override
    protected <T> void onTransactionReady(TransactionIdentifier transaction, Collection<Future<T>> cohortFutures) {
        State localState = this.currentState;
        Preconditions.checkState((boolean)(localState instanceof Allocated), (String)"Readying transaction %s while state is %s", (Object)transaction, (Object)localState);
        TransactionIdentifier currentTx = ((Allocated)localState).getIdentifier();
        Preconditions.checkState((boolean)transaction.equals((Object)currentTx), (String)"Readying transaction %s while %s is allocated", (Object)transaction, (Object)currentTx);
        if (cohortFutures.isEmpty()) {
            this.currentState = IDLE_STATE;
            return;
        }
        Future combined = Futures.sequence(cohortFutures, (ExecutionContext)this.getActorUtils().getClientDispatcher());
        final Submitted newState = new Submitted(transaction, combined);
        this.currentState = newState;
        combined.onComplete((Function1)new OnComplete<Iterable<T>>(){

            public void onComplete(Throwable arg0, Iterable<T> arg1) {
                STATE_UPDATER.compareAndSet(TransactionChainProxy.this, newState, IDLE_STATE);
            }
        }, this.getActorUtils().getClientDispatcher());
    }

    @Override
    protected void onTransactionContextCreated(TransactionIdentifier transactionId) {
        Promise promise = (Promise)this.priorReadOnlyTxPromises.remove(transactionId);
        if (promise != null) {
            promise.success(null);
        }
    }

    private static abstract class DefaultState
    extends State {
        private DefaultState() {
        }

        @Override
        final Future<?> previousFuture() {
            return null;
        }
    }

    private static final class Submitted
    extends Pending {
        Submitted(TransactionIdentifier transaction, Future<?> previousFuture) {
            super(transaction, previousFuture);
        }

        @Override
        void checkReady() {
        }
    }

    private static final class Allocated
    extends Pending {
        Allocated(TransactionIdentifier transaction, Future<?> previousFuture) {
            super(transaction, previousFuture);
        }

        @Override
        void checkReady() {
            throw new IllegalStateException(String.format("Previous transaction %s is not ready yet", this.getIdentifier()));
        }
    }

    private static abstract class Pending
    extends State {
        private final TransactionIdentifier transaction;
        private final Future<?> previousFuture;

        Pending(TransactionIdentifier transaction, Future<?> previousFuture) {
            this.previousFuture = previousFuture;
            this.transaction = Objects.requireNonNull(transaction);
        }

        @Override
        final Future<?> previousFuture() {
            return this.previousFuture;
        }

        final TransactionIdentifier getIdentifier() {
            return this.transaction;
        }
    }

    private static abstract class State {
        private State() {
        }

        abstract void checkReady();

        abstract Future<?> previousFuture();
    }
}

