/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.controller.cluster.io;

import com.google.common.annotations.Beta;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.math.IntMath;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayDeque;
import java.util.Arrays;
import java.util.Deque;
import java.util.Iterator;
import org.opendaylight.controller.cluster.io.ChunkedByteArray;
import org.opendaylight.yangtools.concepts.Either;

@Beta
public final class ChunkedOutputStream
extends OutputStream {
    private static final int MIN_ARRAY_SIZE = 32;
    private final int maxChunkSize;
    private Object result;
    private Deque<byte[]> prevChunks;
    private byte[] currentChunk;
    private int currentOffset;
    private int size;

    public ChunkedOutputStream(int requestedInitialCapacity, int maxChunkSize) {
        Preconditions.checkArgument((boolean)IntMath.isPowerOfTwo((int)maxChunkSize), (String)"Maximum chunk size %s is not a power of two", (int)maxChunkSize);
        Preconditions.checkArgument((maxChunkSize > 0 ? 1 : 0) != 0, (String)"Maximum chunk size %s is not positive", (int)maxChunkSize);
        this.maxChunkSize = maxChunkSize;
        this.currentChunk = new byte[ChunkedOutputStream.initialCapacity(requestedInitialCapacity, maxChunkSize)];
    }

    @Override
    public void write(int b) throws IOException {
        this.checkNotClosed();
        this.ensureOneByte();
        this.currentChunk[this.currentOffset] = (byte)b;
        ++this.currentOffset;
        ++this.size;
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        int count;
        if (len < 0) {
            throw new IndexOutOfBoundsException();
        }
        this.checkNotClosed();
        int fromOffset = off;
        for (int toCopy = len; toCopy != 0; toCopy -= count) {
            count = this.ensureMoreBytes(toCopy);
            System.arraycopy(b, fromOffset, this.currentChunk, this.currentOffset, count);
            this.currentOffset += count;
            this.size += count;
            fromOffset += count;
        }
    }

    @Override
    public void close() {
        if (this.result == null) {
            this.result = this.computeResult();
            this.prevChunks = null;
            this.currentChunk = null;
        }
    }

    public int size() {
        return this.size;
    }

    public Either<byte[], ChunkedByteArray> toVariant() {
        this.checkClosed();
        return this.result instanceof byte[] ? Either.ofFirst((Object)((byte[])this.result)) : Either.ofSecond((Object)new ChunkedByteArray(this.size, (ImmutableList<byte[]>)((ImmutableList)this.result)));
    }

    @VisibleForTesting
    ChunkedByteArray toChunkedByteArray() {
        this.checkClosed();
        return new ChunkedByteArray(this.size, (ImmutableList<byte[]>)(this.result instanceof byte[] ? ImmutableList.of((Object)((byte[])this.result)) : (ImmutableList)this.result));
    }

    private Object computeResult() {
        byte[] chunk;
        if (this.prevChunks == null) {
            return ChunkedOutputStream.trimChunk(this.currentChunk, this.currentOffset);
        }
        if (this.size <= this.maxChunkSize) {
            if (this.currentOffset == 0 && this.prevChunks.size() == 1) {
                return this.prevChunks.getFirst();
            }
            byte[] singleChunk = new byte[this.size];
            int offset = 0;
            for (byte[] chunk2 : this.prevChunks) {
                System.arraycopy(chunk2, 0, singleChunk, offset, chunk2.length);
                offset += chunk2.length;
            }
            System.arraycopy(this.currentChunk, 0, singleChunk, offset, this.currentOffset);
            return singleChunk;
        }
        int headSize = 0;
        int headCount = 0;
        Iterator<byte[]> it = this.prevChunks.iterator();
        while ((chunk = it.next()).length != this.maxChunkSize) {
            headSize += chunk.length;
            ++headCount;
            if (it.hasNext()) continue;
        }
        byte[] head = new byte[headSize];
        int offset = 0;
        for (int i = 0; i < headCount; ++i) {
            byte[] chunk3 = this.prevChunks.removeFirst();
            System.arraycopy(chunk3, 0, head, offset, chunk3.length);
            offset += chunk3.length;
        }
        Verify.verify((offset == head.length ? 1 : 0) != 0);
        this.prevChunks.addFirst(head);
        if (this.currentOffset == 0) {
            return ImmutableList.copyOf(this.prevChunks);
        }
        ImmutableList.Builder builder = ImmutableList.builderWithExpectedSize((int)(this.prevChunks.size() + 1));
        builder.addAll(this.prevChunks);
        builder.add((Object)ChunkedOutputStream.trimChunk(this.currentChunk, this.currentOffset));
        return builder.build();
    }

    private void ensureOneByte() {
        if (this.currentChunk.length == this.currentOffset) {
            this.nextChunk(this.nextChunkSize(this.currentChunk.length));
        }
    }

    private int ensureMoreBytes(int requested) {
        int count;
        int available = this.currentChunk.length - this.currentOffset;
        if (available == 0) {
            this.nextChunk(this.nextChunkSize(this.currentChunk.length, requested));
            available = this.currentChunk.length;
        }
        Verify.verify(((count = Math.min(requested, available)) > 0 ? 1 : 0) != 0);
        return count;
    }

    private void nextChunk(int chunkSize) {
        if (this.prevChunks == null) {
            this.prevChunks = new ArrayDeque<byte[]>();
        }
        this.prevChunks.addLast(this.currentChunk);
        this.currentChunk = new byte[chunkSize];
        this.currentOffset = 0;
    }

    private void checkClosed() {
        Preconditions.checkState((this.result != null ? 1 : 0) != 0, (Object)"Stream has not been closed yet");
    }

    private void checkNotClosed() throws IOException {
        if (this.result != null) {
            throw new IOException("Stream is already closed");
        }
    }

    private int nextChunkSize(int currentSize, int requested) {
        return currentSize == this.maxChunkSize || requested >= this.maxChunkSize ? this.maxChunkSize : Math.max(currentSize * 2, IntMath.ceilingPowerOfTwo((int)requested));
    }

    private int nextChunkSize(int currentSize) {
        return currentSize < this.maxChunkSize ? currentSize * 2 : this.maxChunkSize;
    }

    private static int initialCapacity(int requestedSize, int maxChunkSize) {
        if (requestedSize < 32) {
            return 32;
        }
        if (requestedSize > maxChunkSize) {
            return maxChunkSize;
        }
        return IntMath.ceilingPowerOfTwo((int)requestedSize);
    }

    private static byte[] trimChunk(byte[] chunk, int length) {
        return chunk.length == length ? chunk : Arrays.copyOf(chunk, length);
    }
}

