/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.controller.cluster.messaging;

import akka.actor.ActorRef;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalNotification;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Predicate;
import org.eclipse.jdt.annotation.NonNull;
import org.opendaylight.controller.cluster.io.FileBackedOutputStream;
import org.opendaylight.controller.cluster.io.FileBackedOutputStreamFactory;
import org.opendaylight.controller.cluster.messaging.AbortSlicing;
import org.opendaylight.controller.cluster.messaging.MessageSlice;
import org.opendaylight.controller.cluster.messaging.MessageSliceException;
import org.opendaylight.controller.cluster.messaging.MessageSliceIdentifier;
import org.opendaylight.controller.cluster.messaging.MessageSliceReply;
import org.opendaylight.controller.cluster.messaging.SliceOptions;
import org.opendaylight.controller.cluster.messaging.SlicedMessageState;
import org.opendaylight.yangtools.concepts.Identifier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MessageSlicer
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(MessageSlicer.class);
    private static final AtomicLong SLICER_ID_COUNTER = new AtomicLong(1L);
    public static final int DEFAULT_MAX_SLICING_TRIES = 3;
    private final Cache<MessageSliceIdentifier, SlicedMessageState<ActorRef>> stateCache;
    private final FileBackedOutputStreamFactory fileBackedStreamFactory;
    private final int messageSliceSize;
    private final int maxSlicingTries;
    private final String logContext;
    private final long id;

    MessageSlicer(Builder builder) {
        this.fileBackedStreamFactory = builder.fileBackedStreamFactory;
        this.messageSliceSize = builder.messageSliceSize;
        this.maxSlicingTries = builder.maxSlicingTries;
        this.id = SLICER_ID_COUNTER.getAndIncrement();
        this.logContext = builder.logContext + "_slicer-id-" + this.id;
        CacheBuilder cacheBuilder = CacheBuilder.newBuilder().removalListener(this::stateRemoved);
        if (builder.expireStateAfterInactivityDuration > 0L) {
            cacheBuilder = cacheBuilder.expireAfterAccess(builder.expireStateAfterInactivityDuration, builder.expireStateAfterInactivityUnit);
        }
        this.stateCache = cacheBuilder.build();
    }

    @VisibleForTesting
    long getId() {
        return this.id;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static boolean isHandledMessage(Object message) {
        return message instanceof MessageSliceReply;
    }

    public boolean slice(SliceOptions options) {
        FileBackedOutputStream fileBackedStream;
        block8: {
            Identifier identifier = options.getIdentifier();
            Serializable message = options.getMessage();
            if (message != null) {
                LOG.debug("{}: slice: identifier: {}, message: {}", new Object[]{this.logContext, identifier, message});
                Objects.requireNonNull(this.fileBackedStreamFactory, "The FiledBackedStreamFactory must be set in order to call this slice method");
                fileBackedStream = this.fileBackedStreamFactory.newInstance();
                try (ObjectOutputStream out = new ObjectOutputStream(fileBackedStream);){
                    out.writeObject(message);
                    break block8;
                }
                catch (IOException e) {
                    LOG.debug("{}: Error serializing message for {}", new Object[]{this.logContext, identifier, e});
                    fileBackedStream.cleanup();
                    options.getOnFailureCallback().accept(e);
                    return false;
                }
            }
            fileBackedStream = options.getFileBackedStream();
        }
        return this.initializeSlicing(options, fileBackedStream);
    }

    private boolean initializeSlicing(SliceOptions options, FileBackedOutputStream fileBackedStream) {
        Identifier identifier = options.getIdentifier();
        MessageSliceIdentifier messageSliceId = new MessageSliceIdentifier(identifier, this.id);
        SlicedMessageState<ActorRef> state = null;
        try {
            state = new SlicedMessageState<ActorRef>(messageSliceId, fileBackedStream, this.messageSliceSize, this.maxSlicingTries, options.getReplyTo(), options.getOnFailureCallback(), this.logContext);
            Serializable message = options.getMessage();
            if (state.getTotalSlices() == 1 && message != null) {
                LOG.debug("{}: Message does not need to be sliced - sending original message", (Object)this.logContext);
                state.close();
                MessageSlicer.sendTo(options, message, options.getReplyTo());
                return false;
            }
            MessageSlice firstSlice = MessageSlicer.getNextSliceMessage(state);
            LOG.debug("{}: Sending first slice: {}", (Object)this.logContext, (Object)firstSlice);
            this.stateCache.put((Object)messageSliceId, state);
            MessageSlicer.sendTo(options, firstSlice, ActorRef.noSender());
            return true;
        }
        catch (IOException e) {
            LOG.error("{}: Error initializing SlicedMessageState for {}", new Object[]{this.logContext, identifier, e});
            if (state != null) {
                state.close();
            } else {
                fileBackedStream.cleanup();
            }
            options.getOnFailureCallback().accept(e);
            return false;
        }
    }

    private static void sendTo(SliceOptions options, Object message, ActorRef sender) {
        if (options.getSendToRef() != null) {
            options.getSendToRef().tell(message, sender);
        } else {
            options.getSendToSelection().tell(message, sender);
        }
    }

    public boolean handleMessage(Object message) {
        if (message instanceof MessageSliceReply) {
            MessageSliceReply sliceReply = (MessageSliceReply)message;
            LOG.debug("{}: handleMessage: {}", (Object)this.logContext, (Object)sliceReply);
            return this.onMessageSliceReply(sliceReply);
        }
        return false;
    }

    public void checkExpiredSlicedMessageState() {
        if (this.stateCache.size() > 0L) {
            this.stateCache.cleanUp();
        }
    }

    @Override
    public void close() {
        LOG.debug("{}: Closing", (Object)this.logContext);
        this.stateCache.invalidateAll();
    }

    public void cancelSlicing(@NonNull Predicate<Identifier> filter) {
        this.stateCache.asMap().keySet().removeIf(messageSliceIdentifier -> filter.test(messageSliceIdentifier.getClientIdentifier()));
    }

    private static MessageSlice getNextSliceMessage(SlicedMessageState<ActorRef> state) throws IOException {
        byte[] firstSliceBytes = state.getNextSlice();
        return new MessageSlice(state.getIdentifier(), firstSliceBytes, state.getCurrentSliceIndex(), state.getTotalSlices(), state.getLastSliceHashCode(), state.getReplyTarget());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean onMessageSliceReply(MessageSliceReply reply) {
        MessageSliceIdentifier sliceIdentifier;
        Identifier identifier = reply.getIdentifier();
        if (!(identifier instanceof MessageSliceIdentifier) || (sliceIdentifier = (MessageSliceIdentifier)identifier).getSlicerId() != this.id) {
            return false;
        }
        SlicedMessageState state = (SlicedMessageState)this.stateCache.getIfPresent((Object)identifier);
        if (state == null) {
            LOG.warn("{}: SlicedMessageState not found for {}", (Object)this.logContext, (Object)reply);
            reply.getSendTo().tell((Object)new AbortSlicing(identifier), ActorRef.noSender());
            return true;
        }
        SlicedMessageState slicedMessageState = state;
        synchronized (slicedMessageState) {
            try {
                Optional<MessageSliceException> failure = reply.getFailure();
                if (failure.isPresent()) {
                    LOG.warn("{}: Received failed {}", (Object)this.logContext, (Object)reply);
                    this.processMessageSliceException(failure.get(), state, reply.getSendTo());
                    return true;
                }
                if (state.getCurrentSliceIndex() != reply.getSliceIndex()) {
                    LOG.warn("{}: Slice index {} in {} does not match expected index {}", new Object[]{this.logContext, reply.getSliceIndex(), reply, state.getCurrentSliceIndex()});
                    reply.getSendTo().tell((Object)new AbortSlicing(identifier), ActorRef.noSender());
                    this.possiblyRetrySlicing(state, reply.getSendTo());
                    return true;
                }
                if (state.isLastSlice(reply.getSliceIndex())) {
                    LOG.debug("{}: Received last slice reply for {}", (Object)this.logContext, (Object)identifier);
                    this.removeState(identifier);
                } else {
                    MessageSlice nextSlice = MessageSlicer.getNextSliceMessage(state);
                    LOG.debug("{}: Sending next slice: {}", (Object)this.logContext, (Object)nextSlice);
                    reply.getSendTo().tell((Object)nextSlice, ActorRef.noSender());
                }
            }
            catch (IOException e) {
                LOG.warn("{}: Error processing {}", new Object[]{this.logContext, reply, e});
                this.fail(state, e);
            }
        }
        return true;
    }

    private void processMessageSliceException(MessageSliceException exception, SlicedMessageState<ActorRef> state, ActorRef sendTo) throws IOException {
        if (exception.isRetriable()) {
            this.possiblyRetrySlicing(state, sendTo);
        } else {
            this.fail(state, exception.getCause() != null ? exception.getCause() : exception);
        }
    }

    private void possiblyRetrySlicing(SlicedMessageState<ActorRef> state, ActorRef sendTo) throws IOException {
        if (state.canRetry()) {
            LOG.info("{}: Retrying message slicing for {}", (Object)this.logContext, (Object)state.getIdentifier());
            state.reset();
            sendTo.tell((Object)MessageSlicer.getNextSliceMessage(state), ActorRef.noSender());
        } else {
            String message = String.format("Maximum slicing retries reached for identifier %s - failing the message", state.getIdentifier());
            LOG.warn(message);
            this.fail(state, new RuntimeException(message));
        }
    }

    private void removeState(Identifier identifier) {
        LOG.debug("{}: Removing state for {}", (Object)this.logContext, (Object)identifier);
        this.stateCache.invalidate((Object)identifier);
    }

    private void stateRemoved(RemovalNotification<Identifier, SlicedMessageState<ActorRef>> notification) {
        SlicedMessageState state = (SlicedMessageState)notification.getValue();
        state.close();
        if (notification.wasEvicted()) {
            LOG.warn("{}: SlicedMessageState for {} was expired from the cache", (Object)this.logContext, notification.getKey());
            state.getOnFailureCallback().accept(new RuntimeException(String.format("The slicing state for message identifier %s was expired due to inactivity from the assembling component on the other end", state.getIdentifier())));
        } else {
            LOG.debug("{}: SlicedMessageState for {} was removed from the cache due to {}", new Object[]{this.logContext, notification.getKey(), notification.getCause()});
        }
    }

    private void fail(SlicedMessageState<ActorRef> state, Throwable failure) {
        this.removeState(state.getIdentifier());
        state.getOnFailureCallback().accept(failure);
    }

    @VisibleForTesting
    boolean hasState(Identifier forIdentifier) {
        boolean exists = this.stateCache.getIfPresent((Object)forIdentifier) != null;
        this.stateCache.cleanUp();
        return exists;
    }

    public static class Builder {
        private FileBackedOutputStreamFactory fileBackedStreamFactory;
        private int messageSliceSize = -1;
        private long expireStateAfterInactivityDuration = -1L;
        private TimeUnit expireStateAfterInactivityUnit = TimeUnit.MINUTES;
        private int maxSlicingTries = 3;
        private String logContext = "<no-context>";

        public Builder fileBackedStreamFactory(FileBackedOutputStreamFactory newFileBackedStreamFactory) {
            this.fileBackedStreamFactory = Objects.requireNonNull(newFileBackedStreamFactory);
            return this;
        }

        public Builder messageSliceSize(int newMessageSliceSize) {
            Preconditions.checkArgument((newMessageSliceSize > 0 ? 1 : 0) != 0, (Object)"messageSliceSize must be > 0");
            this.messageSliceSize = newMessageSliceSize;
            return this;
        }

        public Builder maxSlicingTries(int newMaxSlicingTries) {
            Preconditions.checkArgument((newMaxSlicingTries > 0 ? 1 : 0) != 0, (Object)"newMaxSlicingTries must be > 0");
            this.maxSlicingTries = newMaxSlicingTries;
            return this;
        }

        public Builder expireStateAfterInactivity(long duration, TimeUnit unit) {
            Preconditions.checkArgument((duration > 0L ? 1 : 0) != 0, (Object)"duration must be > 0");
            this.expireStateAfterInactivityDuration = duration;
            this.expireStateAfterInactivityUnit = unit;
            return this;
        }

        public Builder logContext(String newLogContext) {
            this.logContext = Objects.requireNonNull(newLogContext);
            return this;
        }

        public MessageSlicer build() {
            return new MessageSlicer(this);
        }
    }
}

