/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.controller.cluster.messaging;

import akka.actor.ActorRef;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalNotification;
import com.google.common.io.ByteSource;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import org.eclipse.jdt.annotation.NonNull;
import org.opendaylight.controller.cluster.io.FileBackedOutputStreamFactory;
import org.opendaylight.controller.cluster.messaging.AbortSlicing;
import org.opendaylight.controller.cluster.messaging.AssembledMessageState;
import org.opendaylight.controller.cluster.messaging.MessageSlice;
import org.opendaylight.controller.cluster.messaging.MessageSliceException;
import org.opendaylight.controller.cluster.messaging.MessageSliceReply;
import org.opendaylight.yangtools.concepts.Identifier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class MessageAssembler
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(MessageAssembler.class);
    private final Cache<Identifier, AssembledMessageState> stateCache;
    private final FileBackedOutputStreamFactory fileBackedStreamFactory;
    private final BiConsumer<Object, ActorRef> assembledMessageCallback;
    private final String logContext;

    MessageAssembler(Builder builder) {
        this.fileBackedStreamFactory = Objects.requireNonNull(builder.fileBackedStreamFactory, "FiledBackedStreamFactory cannot be null");
        this.assembledMessageCallback = Objects.requireNonNull(builder.assembledMessageCallback, "assembledMessageCallback cannot be null");
        this.logContext = builder.logContext;
        this.stateCache = CacheBuilder.newBuilder().expireAfterAccess(builder.expireStateAfterInactivityDuration, builder.expireStateAfterInactivityUnit).removalListener(this::stateRemoved).build();
    }

    public static Builder builder() {
        return new Builder();
    }

    public static boolean isHandledMessage(Object message) {
        return message instanceof MessageSlice || message instanceof AbortSlicing;
    }

    @Override
    public void close() {
        LOG.debug("{}: Closing", (Object)this.logContext);
        this.stateCache.invalidateAll();
    }

    public void checkExpiredAssembledMessageState() {
        if (this.stateCache.size() > 0L) {
            this.stateCache.cleanUp();
        }
    }

    public boolean handleMessage(Object message, @NonNull ActorRef sendTo) {
        if (message instanceof MessageSlice) {
            MessageSlice messageSlice = (MessageSlice)message;
            LOG.debug("{}: handleMessage: {}", (Object)this.logContext, (Object)messageSlice);
            this.onMessageSlice(messageSlice, sendTo);
            return true;
        }
        if (message instanceof AbortSlicing) {
            AbortSlicing abortSlicing = (AbortSlicing)message;
            LOG.debug("{}: handleMessage: {}", (Object)this.logContext, (Object)abortSlicing);
            this.onAbortSlicing(abortSlicing);
            return true;
        }
        return false;
    }

    private void onMessageSlice(MessageSlice messageSlice, ActorRef sendTo) {
        Identifier identifier = messageSlice.getIdentifier();
        try {
            AssembledMessageState state = (AssembledMessageState)this.stateCache.get((Object)identifier, () -> this.createState(messageSlice));
            this.processMessageSliceForState(messageSlice, state, sendTo);
        }
        catch (ExecutionException e) {
            MessageSliceException sliceEx;
            Throwable cause = e.getCause();
            MessageSliceException messageSliceEx = cause instanceof MessageSliceException ? (sliceEx = (MessageSliceException)cause) : new MessageSliceException(String.format("Error creating state for identifier %s", identifier), cause);
            messageSlice.getReplyTo().tell((Object)MessageSliceReply.failed(identifier, messageSliceEx, sendTo), ActorRef.noSender());
        }
    }

    private AssembledMessageState createState(MessageSlice messageSlice) throws MessageSliceException {
        Identifier identifier = messageSlice.getIdentifier();
        if (messageSlice.getSliceIndex() == 1) {
            LOG.debug("{}: Received first slice for {} - creating AssembledMessageState", (Object)this.logContext, (Object)identifier);
            return new AssembledMessageState(identifier, messageSlice.getTotalSlices(), this.fileBackedStreamFactory, this.logContext);
        }
        LOG.debug("{}: AssembledMessageState not found for {} - returning failed reply", (Object)this.logContext, (Object)identifier);
        throw new MessageSliceException(String.format("No assembled state found for identifier %s and slice index %s", identifier, messageSlice.getSliceIndex()), true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processMessageSliceForState(MessageSlice messageSlice, AssembledMessageState state, ActorRef sendTo) {
        Identifier identifier = messageSlice.getIdentifier();
        ActorRef replyTo = messageSlice.getReplyTo();
        Object reAssembledMessage = null;
        AssembledMessageState assembledMessageState = state;
        synchronized (assembledMessageState) {
            int sliceIndex = messageSlice.getSliceIndex();
            try {
                MessageSliceReply successReply = MessageSliceReply.success(identifier, sliceIndex, sendTo);
                if (state.addSlice(sliceIndex, messageSlice.getData(), messageSlice.getLastSliceHashCode())) {
                    LOG.debug("{}: Received last slice for {}", (Object)this.logContext, (Object)identifier);
                    reAssembledMessage = MessageAssembler.reAssembleMessage(state);
                    replyTo.tell((Object)successReply, ActorRef.noSender());
                    this.removeState(identifier);
                } else {
                    LOG.debug("{}: Added slice for {} - expecting more", (Object)this.logContext, (Object)identifier);
                    replyTo.tell((Object)successReply, ActorRef.noSender());
                }
            }
            catch (MessageSliceException e) {
                LOG.warn("{}: Error processing {}", new Object[]{this.logContext, messageSlice, e});
                replyTo.tell((Object)MessageSliceReply.failed(identifier, e, sendTo), ActorRef.noSender());
                this.removeState(identifier);
            }
        }
        if (reAssembledMessage != null) {
            LOG.debug("{}: Notifying callback of re-assembled message {}", (Object)this.logContext, reAssembledMessage);
            this.assembledMessageCallback.accept(reAssembledMessage, replyTo);
        }
    }

    private static Object reAssembleMessage(AssembledMessageState state) throws MessageSliceException {
        Object object;
        ByteSource assembledBytes = state.getAssembledBytes();
        ObjectInputStream in = new ObjectInputStream(assembledBytes.openStream());
        try {
            object = in.readObject();
        }
        catch (Throwable throwable) {
            try {
                try {
                    in.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException | ClassNotFoundException e) {
                throw new MessageSliceException(String.format("Error re-assembling bytes for identifier %s", state.getIdentifier()), e);
            }
        }
        in.close();
        return object;
    }

    private void onAbortSlicing(AbortSlicing message) {
        this.removeState(message.getIdentifier());
    }

    private void removeState(Identifier identifier) {
        LOG.debug("{}: Removing state for {}", (Object)this.logContext, (Object)identifier);
        this.stateCache.invalidate((Object)identifier);
    }

    private void stateRemoved(RemovalNotification<Identifier, AssembledMessageState> notification) {
        if (notification.wasEvicted()) {
            LOG.warn("{}: AssembledMessageState for {} was expired from the cache", (Object)this.logContext, notification.getKey());
        } else {
            LOG.debug("{}: AssembledMessageState for {} was removed from the cache due to {}", new Object[]{this.logContext, notification.getKey(), notification.getCause()});
        }
        ((AssembledMessageState)notification.getValue()).close();
    }

    @VisibleForTesting
    boolean hasState(Identifier forIdentifier) {
        boolean exists = this.stateCache.getIfPresent((Object)forIdentifier) != null;
        this.stateCache.cleanUp();
        return exists;
    }

    public static class Builder {
        private FileBackedOutputStreamFactory fileBackedStreamFactory;
        private BiConsumer<Object, ActorRef> assembledMessageCallback;
        private long expireStateAfterInactivityDuration = 1L;
        private TimeUnit expireStateAfterInactivityUnit = TimeUnit.MINUTES;
        private String logContext = "<no-context>";

        public Builder fileBackedStreamFactory(FileBackedOutputStreamFactory newFileBackedStreamFactory) {
            this.fileBackedStreamFactory = Objects.requireNonNull(newFileBackedStreamFactory);
            return this;
        }

        public Builder assembledMessageCallback(BiConsumer<Object, ActorRef> newAssembledMessageCallback) {
            this.assembledMessageCallback = newAssembledMessageCallback;
            return this;
        }

        public Builder expireStateAfterInactivity(long duration, TimeUnit unit) {
            Preconditions.checkArgument((duration > 0L ? 1 : 0) != 0, (Object)"duration must be > 0");
            this.expireStateAfterInactivityDuration = duration;
            this.expireStateAfterInactivityUnit = unit;
            return this;
        }

        public Builder logContext(String newLogContext) {
            this.logContext = newLogContext;
            return this;
        }

        public MessageAssembler build() {
            return new MessageAssembler(this);
        }
    }
}

