/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.yangtools.yang.model.repo.fs;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.FluentFuture;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.opendaylight.yangtools.util.concurrent.FluentFutures;
import org.opendaylight.yangtools.yang.common.Revision;
import org.opendaylight.yangtools.yang.model.repo.api.MissingSchemaSourceException;
import org.opendaylight.yangtools.yang.model.repo.api.SchemaSourceRepresentation;
import org.opendaylight.yangtools.yang.model.repo.api.SourceIdentifier;
import org.opendaylight.yangtools.yang.model.repo.api.YangTextSchemaSource;
import org.opendaylight.yangtools.yang.model.repo.spi.AbstractSchemaSourceCache;
import org.opendaylight.yangtools.yang.model.repo.spi.PotentialSchemaSource;
import org.opendaylight.yangtools.yang.model.repo.spi.SchemaSourceRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class FilesystemSchemaSourceCache<T extends SchemaSourceRepresentation>
extends AbstractSchemaSourceCache<T> {
    private static final Logger LOG = LoggerFactory.getLogger(FilesystemSchemaSourceCache.class);
    private static final Map<Class<? extends SchemaSourceRepresentation>, StorageAdapter<? extends SchemaSourceRepresentation>> STORAGE_ADAPTERS = Collections.singletonMap(YangTextSchemaSource.class, new YangTextSchemaStorageAdapter());
    private static final Pattern CACHED_FILE_PATTERN = Pattern.compile("(?<moduleName>[^@]+)(@(?<revision>" + Revision.STRING_FORMAT_PATTERN + "))?");
    private final Class<T> representation;
    private final File storageDirectory;

    public FilesystemSchemaSourceCache(SchemaSourceRegistry consumer, Class<T> representation, File storageDirectory) {
        super(consumer, representation, PotentialSchemaSource.Costs.LOCAL_IO);
        this.representation = representation;
        this.storageDirectory = Objects.requireNonNull(storageDirectory);
        FilesystemSchemaSourceCache.checkSupportedRepresentation(representation);
        Preconditions.checkArgument((storageDirectory.mkdirs() || storageDirectory.isDirectory() ? 1 : 0) != 0, (String)"Unable to create cache directory at %s", (Object)storageDirectory);
        Preconditions.checkArgument((boolean)storageDirectory.canWrite());
        Preconditions.checkArgument((boolean)storageDirectory.canRead());
        this.init();
    }

    private static void checkSupportedRepresentation(Class<? extends SchemaSourceRepresentation> representation) {
        for (Class<? extends SchemaSourceRepresentation> supportedRepresentation : STORAGE_ADAPTERS.keySet()) {
            if (!supportedRepresentation.isAssignableFrom(representation)) continue;
            return;
        }
        throw new IllegalArgumentException(String.format("This cache does not support representation: %s, supported representations are: %s", representation, STORAGE_ADAPTERS.keySet()));
    }

    private void init() {
        CachedModulesFileVisitor fileVisitor = new CachedModulesFileVisitor();
        try {
            Files.walkFileTree(this.storageDirectory.toPath(), fileVisitor);
        }
        catch (IOException e) {
            LOG.warn("Unable to restore cache from {}. Starting with an empty cache", (Object)this.storageDirectory, (Object)e);
            return;
        }
        fileVisitor.getCachedSchemas().stream().forEach(x$0 -> this.register((SourceIdentifier)x$0));
    }

    public synchronized FluentFuture<? extends T> getSource(SourceIdentifier sourceIdentifier) {
        File file = FilesystemSchemaSourceCache.sourceIdToFile(sourceIdentifier, this.storageDirectory);
        if (file.exists() && file.canRead()) {
            LOG.trace("Source {} found in cache as {}", (Object)sourceIdentifier, (Object)file);
            SchemaSourceRepresentation restored = STORAGE_ADAPTERS.get(this.representation).restore(sourceIdentifier, file);
            return FluentFutures.immediateFluentFuture((Object)((SchemaSourceRepresentation)this.representation.cast(restored)));
        }
        LOG.debug("Source {} not found in cache as {}", (Object)sourceIdentifier, (Object)file);
        return FluentFutures.immediateFailedFluentFuture((Throwable)new MissingSchemaSourceException("Source not found", sourceIdentifier));
    }

    protected synchronized void offer(T source) {
        LOG.trace("Source {} offered to cache", (Object)source.getIdentifier());
        File file = this.sourceIdToFile(source);
        if (file.exists()) {
            LOG.debug("Source {} already in cache as {}", (Object)source.getIdentifier(), (Object)file);
            return;
        }
        this.storeSource(file, source);
        this.register(source.getIdentifier());
        LOG.trace("Source {} stored in cache as {}", (Object)source.getIdentifier(), (Object)file);
    }

    private File sourceIdToFile(T source) {
        return FilesystemSchemaSourceCache.sourceIdToFile(source.getIdentifier(), this.storageDirectory);
    }

    static File sourceIdToFile(SourceIdentifier identifier, File storageDirectory) {
        Revision rev = identifier.revision();
        File file = rev == null ? FilesystemSchemaSourceCache.findFileWithNewestRev(identifier, storageDirectory) : new File(storageDirectory, identifier.toYangFilename());
        return file;
    }

    private static File findFileWithNewestRev(final SourceIdentifier identifier, File storageDirectory) {
        File[] files = storageDirectory.listFiles(new FilenameFilter(){
            final Pattern pat;
            {
                this.pat = Pattern.compile(Pattern.quote(identifier.name().getLocalName()) + "(\\.yang|@\\d\\d\\d\\d-\\d\\d-\\d\\d.yang)");
            }

            @Override
            public boolean accept(File dir, String name) {
                return this.pat.matcher(name).matches();
            }
        });
        if (files.length == 0) {
            return new File(storageDirectory, identifier.toYangFilename());
        }
        if (files.length == 1) {
            return files[0];
        }
        File file = null;
        TreeMap map = new TreeMap(Revision::compare);
        for (File sorted : files) {
            String fileName = sorted.getName();
            Matcher match = Revision.STRING_FORMAT_PATTERN.matcher(fileName);
            if (match.find()) {
                Revision rev;
                String revStr = match.group();
                try {
                    rev = Revision.of((String)revStr);
                }
                catch (DateTimeParseException e) {
                    LOG.info("Unable to parse date from yang file name {}, falling back to not-present", (Object)fileName, (Object)e);
                    rev = null;
                }
                map.put(Optional.ofNullable(rev), sorted);
                continue;
            }
            map.put(Optional.empty(), sorted);
        }
        file = (File)map.lastEntry().getValue();
        return file;
    }

    private void storeSource(File file, T schemaRepresentation) {
        STORAGE_ADAPTERS.get(this.representation).store(file, (SchemaSourceRepresentation)schemaRepresentation);
    }

    private static final class CachedModulesFileVisitor
    extends SimpleFileVisitor<Path> {
        private final List<SourceIdentifier> cachedSchemas = new ArrayList<SourceIdentifier>();

        private CachedModulesFileVisitor() {
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            FileVisitResult fileVisitResult = super.visitFile(file, attrs);
            String fileName = file.toFile().getName();
            Optional<SourceIdentifier> si = CachedModulesFileVisitor.getSourceIdentifier(fileName = com.google.common.io.Files.getNameWithoutExtension((String)fileName));
            if (si.isPresent()) {
                LOG.trace("Restoring cached file {} as {}", (Object)file, (Object)si.orElseThrow());
                this.cachedSchemas.add(si.orElseThrow());
            } else {
                LOG.debug("Skipping cached file {}, cannot restore source identifier from filename: {}, does not match {}", new Object[]{file, fileName, CACHED_FILE_PATTERN});
            }
            return fileVisitResult;
        }

        private static Optional<SourceIdentifier> getSourceIdentifier(String fileName) {
            Matcher matcher = CACHED_FILE_PATTERN.matcher(fileName);
            if (matcher.matches()) {
                String moduleName = matcher.group("moduleName");
                String revision = matcher.group("revision");
                return Optional.of(new SourceIdentifier(moduleName, revision));
            }
            return Optional.empty();
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
            LOG.warn("Unable to restore cached file {}. Ignoring", (Object)file, (Object)exc);
            return FileVisitResult.CONTINUE;
        }

        public List<SourceIdentifier> getCachedSchemas() {
            return this.cachedSchemas;
        }
    }

    private static abstract class StorageAdapter<T extends SchemaSourceRepresentation> {
        private final Class<T> supportedType;

        protected StorageAdapter(Class<T> supportedType) {
            this.supportedType = supportedType;
        }

        void store(File file, SchemaSourceRepresentation schemaSourceRepresentation) {
            Preconditions.checkArgument((boolean)this.supportedType.isAssignableFrom(schemaSourceRepresentation.getClass()), (String)"Cannot store schema source %s, this adapter only supports %s", (Object)schemaSourceRepresentation, this.supportedType);
            this.storeAsType(file, (SchemaSourceRepresentation)this.supportedType.cast(schemaSourceRepresentation));
        }

        protected abstract void storeAsType(File var1, T var2);

        T restore(SourceIdentifier sourceIdentifier, File cachedSource) {
            Preconditions.checkArgument((boolean)cachedSource.isFile());
            Preconditions.checkArgument((boolean)cachedSource.exists());
            Preconditions.checkArgument((boolean)cachedSource.canRead());
            return this.restoreAsType(sourceIdentifier, cachedSource);
        }

        abstract T restoreAsType(SourceIdentifier var1, File var2);
    }

    private static final class YangTextSchemaStorageAdapter
    extends StorageAdapter<YangTextSchemaSource> {
        protected YangTextSchemaStorageAdapter() {
            super(YangTextSchemaSource.class);
        }

        @Override
        @SuppressFBWarnings(value={"RCN_REDUNDANT_NULLCHECK_WOULD_HAVE_BEEN_A_NPE"}, justification="https://github.com/spotbugs/spotbugs/issues/600")
        protected void storeAsType(File file, YangTextSchemaSource cast) {
            try (InputStream castStream = cast.openStream();){
                Files.copy(castStream, file.toPath(), StandardCopyOption.REPLACE_EXISTING);
            }
            catch (IOException e) {
                throw new IllegalStateException("Cannot store schema source " + cast.getIdentifier() + " to " + file, e);
            }
        }

        @Override
        YangTextSchemaSource restoreAsType(SourceIdentifier sourceIdentifier, File cachedSource) {
            return YangTextSchemaSource.forPath((Path)cachedSource.toPath(), (SourceIdentifier)sourceIdentifier);
        }
    }
}

