package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.mdsaltrace.rev160908;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableMap;
import java.lang.Boolean;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link ConfigBuilder} instances.
 *
 * @see ConfigBuilder
 *
 */
public class ConfigBuilder implements Builder<Config> {

    private List<String> _registrationWatches;
    private List<String> _writeWatches;
    private Boolean _transactionDebugContextEnabled;


    Map<Class<? extends Augmentation<Config>>, Augmentation<Config>> augmentation = Collections.emptyMap();

    public ConfigBuilder() {
    }

    public ConfigBuilder(Config base) {
        this._registrationWatches = base.getRegistrationWatches();
        this._writeWatches = base.getWriteWatches();
        this._transactionDebugContextEnabled = base.isTransactionDebugContextEnabled();
        if (base instanceof ConfigImpl) {
            ConfigImpl impl = (ConfigImpl) base;
            if (!impl.augmentation.isEmpty()) {
                this.augmentation = new HashMap<>(impl.augmentation);
            }
        } else if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<Config>>, Augmentation<Config>> aug =((AugmentationHolder<Config>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
    }


    public List<String> getRegistrationWatches() {
        return _registrationWatches;
    }
    
    public List<String> getWriteWatches() {
        return _writeWatches;
    }
    
    public Boolean isTransactionDebugContextEnabled() {
        return _transactionDebugContextEnabled;
    }
    
    @SuppressWarnings("unchecked")
    public <E extends Augmentation<Config>> E augmentation(Class<E> augmentationType) {
        return (E) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    public ConfigBuilder setRegistrationWatches(final List<String> values) {
        this._registrationWatches = values;
        return this;
    }
    
    public ConfigBuilder setWriteWatches(final List<String> values) {
        this._writeWatches = values;
        return this;
    }
    
    
    public ConfigBuilder setTransactionDebugContextEnabled(final Boolean value) {
        this._transactionDebugContextEnabled = value;
        return this;
    }
    
    public ConfigBuilder addAugmentation(Class<? extends Augmentation<Config>> augmentationType, Augmentation<Config> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public ConfigBuilder removeAugmentation(Class<? extends Augmentation<Config>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public Config build() {
        return new ConfigImpl(this);
    }

    private static final class ConfigImpl implements Config {
    
        private final List<String> _registrationWatches;
        private final List<String> _writeWatches;
        private final Boolean _transactionDebugContextEnabled;
    
        private Map<Class<? extends Augmentation<Config>>, Augmentation<Config>> augmentation = Collections.emptyMap();
    
        ConfigImpl(ConfigBuilder base) {
            this._registrationWatches = base.getRegistrationWatches();
            this._writeWatches = base.getWriteWatches();
            this._transactionDebugContextEnabled = base.isTransactionDebugContextEnabled();
            this.augmentation = ImmutableMap.copyOf(base.augmentation);
        }
    
        @Override
        public Class<Config> getImplementedInterface() {
            return Config.class;
        }
    
        @Override
        public List<String> getRegistrationWatches() {
            return _registrationWatches;
        }
        
        @Override
        public List<String> getWriteWatches() {
            return _writeWatches;
        }
        
        @Override
        public Boolean isTransactionDebugContextEnabled() {
            return _transactionDebugContextEnabled;
        }
        
        @SuppressWarnings("unchecked")
        @Override
        public <E extends Augmentation<Config>> E augmentation(Class<E> augmentationType) {
            return (E) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_registrationWatches);
            result = prime * result + Objects.hashCode(_writeWatches);
            result = prime * result + Objects.hashCode(_transactionDebugContextEnabled);
            result = prime * result + Objects.hashCode(augmentation);
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!Config.class.equals(((DataObject)obj).getImplementedInterface())) {
                return false;
            }
            Config other = (Config)obj;
            if (!Objects.equals(_registrationWatches, other.getRegistrationWatches())) {
                return false;
            }
            if (!Objects.equals(_writeWatches, other.getWriteWatches())) {
                return false;
            }
            if (!Objects.equals(_transactionDebugContextEnabled, other.isTransactionDebugContextEnabled())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                ConfigImpl otherImpl = (ConfigImpl) obj;
                if (!Objects.equals(augmentation, otherImpl.augmentation)) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<Config>>, Augmentation<Config>> e : augmentation.entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("Config");
            CodeHelpers.appendValue(helper, "_registrationWatches", _registrationWatches);
            CodeHelpers.appendValue(helper, "_writeWatches", _writeWatches);
            CodeHelpers.appendValue(helper, "_transactionDebugContextEnabled", _transactionDebugContextEnabled);
            CodeHelpers.appendValue(helper, "augmentation", augmentation.values());
            return helper.toString();
        }
    }
}
