package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.controller.md.sal.cluster.admin.rev151013;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Deprecated;
import java.lang.Long;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;
import org.opendaylight.yangtools.yang.common.Uint32;

/**
 * Class that builds {@link BackupDatastoreInputBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     BackupDatastoreInputBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new BackupDatastoreInputBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of BackupDatastoreInputBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see BackupDatastoreInputBuilder
 * @see Builder
 *
 */
public class BackupDatastoreInputBuilder implements Builder<BackupDatastoreInput> {

    private String _filePath;
    private Uint32 _timeout;


    Map<Class<? extends Augmentation<BackupDatastoreInput>>, Augmentation<BackupDatastoreInput>> augmentation = Collections.emptyMap();

    public BackupDatastoreInputBuilder() {
    }

    public BackupDatastoreInputBuilder(BackupDatastoreInput base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<BackupDatastoreInput>>, Augmentation<BackupDatastoreInput>> aug =((AugmentationHolder<BackupDatastoreInput>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._filePath = base.getFilePath();
        this._timeout = base.getTimeout();
    }


    public String getFilePath() {
        return _filePath;
    }
    
    public Uint32 getTimeout() {
        return _timeout;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<BackupDatastoreInput>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(Objects.requireNonNull(augmentationType));
    }

    
    public BackupDatastoreInputBuilder setFilePath(final String value) {
        this._filePath = value;
        return this;
    }
    
    private static void checkTimeoutRange(final long value) {
        if (value >= 1L) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[1..4294967295]]", value);
    }
    
    public BackupDatastoreInputBuilder setTimeout(final Uint32 value) {
        if (value != null) {
            checkTimeoutRange(value.longValue());
            
        }
        this._timeout = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setTimeout(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public BackupDatastoreInputBuilder setTimeout(final Long value) {
        return setTimeout(CodeHelpers.compatUint(value));
    }
    
    /**
      * Add an augmentation to this builder's product.
      *
      * @param augmentation augmentation to be added
      * @return this builder
      * @throws NullPointerException if {@code augmentation} is null
      */
    public BackupDatastoreInputBuilder addAugmentation(Augmentation<BackupDatastoreInput> augmentation) {
        return doAddAugmentation(augmentation.implementedInterface(), augmentation);
    }
    
    /**
      * Add or remove an augmentation to this builder's product.
      *
      * @param augmentationType augmentation type to be added or removed
      * @param augmentationValue augmentation value, null if the augmentation type should be removed
      * @return this builder
      * @deprecated Use either {@link #addAugmentation(Augmentation)} or {@link #removeAugmentation(Class)} instead.
      */
    @Deprecated(forRemoval = true)
    public BackupDatastoreInputBuilder addAugmentation(Class<? extends Augmentation<BackupDatastoreInput>> augmentationType, Augmentation<BackupDatastoreInput> augmentationValue) {
        return augmentationValue == null ? removeAugmentation(augmentationType) : doAddAugmentation(augmentationType, augmentationValue);
    }
    
    /**
      * Remove an augmentation from this builder's product. If this builder does not track such an augmentation
      * type, this method does nothing.
      *
      * @param augmentationType augmentation type to be removed
      * @return this builder
      */
    public BackupDatastoreInputBuilder removeAugmentation(Class<? extends Augmentation<BackupDatastoreInput>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }
    
    private BackupDatastoreInputBuilder doAddAugmentation(Class<? extends Augmentation<BackupDatastoreInput>> augmentationType, Augmentation<BackupDatastoreInput> augmentationValue) {
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }

    @Override
    public BackupDatastoreInput build() {
        return new BackupDatastoreInputImpl(this);
    }

    private static final class BackupDatastoreInputImpl
        extends AbstractAugmentable<BackupDatastoreInput>
        implements BackupDatastoreInput {
    
        private final String _filePath;
        private final Uint32 _timeout;
    
        BackupDatastoreInputImpl(BackupDatastoreInputBuilder base) {
            super(base.augmentation);
            this._filePath = base.getFilePath();
            this._timeout = base.getTimeout();
        }
    
        @Override
        public String getFilePath() {
            return _filePath;
        }
        
        @Override
        public Uint32 getTimeout() {
            return _timeout;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_filePath);
            result = prime * result + Objects.hashCode(_timeout);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!BackupDatastoreInput.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            BackupDatastoreInput other = (BackupDatastoreInput)obj;
            if (!Objects.equals(_filePath, other.getFilePath())) {
                return false;
            }
            if (!Objects.equals(_timeout, other.getTimeout())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                BackupDatastoreInputImpl otherImpl = (BackupDatastoreInputImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<BackupDatastoreInput>>, Augmentation<BackupDatastoreInput>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("BackupDatastoreInput");
            CodeHelpers.appendValue(helper, "_filePath", _filePath);
            CodeHelpers.appendValue(helper, "_timeout", _timeout);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
