/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.controller.cluster.datastore.utils;

import com.codahale.metrics.Snapshot;
import com.codahale.metrics.Timer;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.RateLimiter;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.opendaylight.controller.cluster.datastore.DatastoreContext;
import org.opendaylight.controller.cluster.datastore.utils.ActorContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TransactionRateLimiter {
    private static final Logger LOG = LoggerFactory.getLogger(TransactionRateLimiter.class);
    private final ActorContext actorContext;
    private final long commitTimeoutInSeconds;
    private final String dataStoreName;
    private final RateLimiter txRateLimiter;
    private final AtomicLong acquireCount = new AtomicLong();
    private volatile long pollOnCount = 1L;

    public TransactionRateLimiter(ActorContext actorContext) {
        this.actorContext = actorContext;
        this.commitTimeoutInSeconds = actorContext.getDatastoreContext().getShardTransactionCommitTimeoutInSeconds();
        this.dataStoreName = actorContext.getDataStoreName();
        this.txRateLimiter = RateLimiter.create((double)actorContext.getDatastoreContext().getTransactionCreationInitialRateLimit());
    }

    public void acquire() {
        this.adjustRateLimit();
        this.txRateLimiter.acquire();
    }

    private void adjustRateLimit() {
        long count = this.acquireCount.incrementAndGet();
        if (count >= this.pollOnCount) {
            Timer commitTimer = this.actorContext.getOperationTimer("commit");
            double newRateLimit = TransactionRateLimiter.calculateNewRateLimit(commitTimer, this.commitTimeoutInSeconds);
            if (newRateLimit < 1.0) {
                newRateLimit = this.getRateLimitFromOtherDataStores();
            }
            if (newRateLimit >= 1.0) {
                this.txRateLimiter.setRate(newRateLimit);
                this.pollOnCount = count + (long)newRateLimit / 2L;
            }
        }
    }

    public double getTxCreationLimit() {
        return this.txRateLimiter.getRate();
    }

    private double getRateLimitFromOtherDataStores() {
        for (String name : DatastoreContext.getGlobalDatastoreNames()) {
            double newRateLimit;
            if (name.equals(this.dataStoreName) || !((newRateLimit = TransactionRateLimiter.calculateNewRateLimit(this.actorContext.getOperationTimer(name, "commit"), this.commitTimeoutInSeconds)) > 0.0)) continue;
            LOG.debug("On unused Tx - data Store {} commit rateLimit adjusted to {}", (Object)this.dataStoreName, (Object)newRateLimit);
            return newRateLimit;
        }
        return -1.0;
    }

    private static double calculateNewRateLimit(Timer commitTimer, long commitTimeoutInSeconds) {
        if (commitTimer == null) {
            return 0.0;
        }
        Snapshot timerSnapshot = commitTimer.getSnapshot();
        double newRateLimit = 0.0;
        long commitTimeoutInNanos = TimeUnit.SECONDS.toNanos(commitTimeoutInSeconds);
        for (int i = 1; i <= 10; ++i) {
            double percentileTimeInNanos = timerSnapshot.getValue((double)i * 0.1);
            if (!(percentileTimeInNanos > 0.0)) continue;
            double percentileRateLimit = (double)commitTimeoutInNanos / percentileTimeInNanos;
            newRateLimit += percentileRateLimit;
        }
        return newRateLimit / (double)(commitTimeoutInSeconds * 10L);
    }

    @VisibleForTesting
    long getPollOnCount() {
        return this.pollOnCount;
    }

    @VisibleForTesting
    void setPollOnCount(long value) {
        this.pollOnCount = value;
    }

    @VisibleForTesting
    void setAcquireCount(long value) {
        this.acquireCount.set(value);
    }
}

