/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.controller.cluster.datastore;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import javax.annotation.Nullable;
import javax.annotation.concurrent.NotThreadSafe;
import org.opendaylight.controller.cluster.access.commands.CreateLocalHistoryRequest;
import org.opendaylight.controller.cluster.access.commands.DeadHistoryException;
import org.opendaylight.controller.cluster.access.commands.DestroyLocalHistoryRequest;
import org.opendaylight.controller.cluster.access.commands.LocalHistoryRequest;
import org.opendaylight.controller.cluster.access.commands.LocalHistorySuccess;
import org.opendaylight.controller.cluster.access.commands.OutOfSequenceEnvelopeException;
import org.opendaylight.controller.cluster.access.commands.PurgeLocalHistoryRequest;
import org.opendaylight.controller.cluster.access.commands.TransactionRequest;
import org.opendaylight.controller.cluster.access.commands.TransactionSuccess;
import org.opendaylight.controller.cluster.access.commands.UnknownHistoryException;
import org.opendaylight.controller.cluster.access.concepts.ClientIdentifier;
import org.opendaylight.controller.cluster.access.concepts.LocalHistoryIdentifier;
import org.opendaylight.controller.cluster.access.concepts.RequestEnvelope;
import org.opendaylight.controller.cluster.access.concepts.RequestException;
import org.opendaylight.controller.cluster.access.concepts.RequestSuccess;
import org.opendaylight.controller.cluster.access.concepts.TransactionIdentifier;
import org.opendaylight.controller.cluster.access.concepts.UnsupportedRequestException;
import org.opendaylight.controller.cluster.datastore.AbstractFrontendHistory;
import org.opendaylight.controller.cluster.datastore.LocalFrontendHistory;
import org.opendaylight.controller.cluster.datastore.ShardDataTree;
import org.opendaylight.controller.cluster.datastore.ShardDataTreeCohort;
import org.opendaylight.controller.cluster.datastore.ShardDataTreeTransactionChain;
import org.opendaylight.controller.cluster.datastore.SimpleShardDataTreeCohort;
import org.opendaylight.controller.cluster.datastore.StandaloneFrontendHistory;
import org.opendaylight.controller.cluster.datastore.utils.UnsignedLongRangeSet;
import org.opendaylight.yangtools.concepts.Identifiable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
final class LeaderFrontendState
implements Identifiable<ClientIdentifier> {
    private static final Logger LOG = LoggerFactory.getLogger(LeaderFrontendState.class);
    private final Map<LocalHistoryIdentifier, LocalFrontendHistory> localHistories;
    private final UnsignedLongRangeSet purgedHistories;
    private final AbstractFrontendHistory standaloneHistory;
    private final ShardDataTree tree;
    private final ClientIdentifier clientId;
    private final String persistenceId;
    private long lastConnectTicks;
    private long lastSeenTicks;
    private long expectedTxSequence;
    private Long lastSeenHistory = null;

    LeaderFrontendState(String persistenceId, ClientIdentifier clientId, ShardDataTree tree) {
        this(persistenceId, clientId, tree, UnsignedLongRangeSet.create(), StandaloneFrontendHistory.create(persistenceId, clientId, tree), new HashMap<LocalHistoryIdentifier, LocalFrontendHistory>());
    }

    LeaderFrontendState(String persistenceId, ClientIdentifier clientId, ShardDataTree tree, UnsignedLongRangeSet purgedHistories, AbstractFrontendHistory standaloneHistory, Map<LocalHistoryIdentifier, LocalFrontendHistory> localHistories) {
        this.persistenceId = (String)Preconditions.checkNotNull((Object)persistenceId);
        this.clientId = (ClientIdentifier)Preconditions.checkNotNull((Object)clientId);
        this.tree = (ShardDataTree)Preconditions.checkNotNull((Object)tree);
        this.purgedHistories = (UnsignedLongRangeSet)Preconditions.checkNotNull((Object)purgedHistories);
        this.standaloneHistory = (AbstractFrontendHistory)Preconditions.checkNotNull((Object)standaloneHistory);
        this.localHistories = (Map)Preconditions.checkNotNull(localHistories);
        this.lastSeenTicks = tree.readTime();
    }

    public ClientIdentifier getIdentifier() {
        return this.clientId;
    }

    private void checkRequestSequence(RequestEnvelope envelope) throws OutOfSequenceEnvelopeException {
        if (this.expectedTxSequence != envelope.getTxSequence()) {
            throw new OutOfSequenceEnvelopeException(this.expectedTxSequence);
        }
    }

    private void expectNextRequest() {
        ++this.expectedTxSequence;
    }

    @Nullable
    LocalHistorySuccess handleLocalHistoryRequest(LocalHistoryRequest<?> request, RequestEnvelope envelope, long now) throws RequestException {
        this.checkRequestSequence(envelope);
        try {
            if (request instanceof CreateLocalHistoryRequest) {
                LocalHistorySuccess localHistorySuccess = this.handleCreateHistory((CreateLocalHistoryRequest)request, envelope, now);
                return localHistorySuccess;
            }
            if (request instanceof DestroyLocalHistoryRequest) {
                LocalHistorySuccess localHistorySuccess = this.handleDestroyHistory((DestroyLocalHistoryRequest)request, envelope, now);
                return localHistorySuccess;
            }
            if (request instanceof PurgeLocalHistoryRequest) {
                LocalHistorySuccess localHistorySuccess = this.handlePurgeHistory((PurgeLocalHistoryRequest)request, envelope, now);
                return localHistorySuccess;
            }
            LOG.warn("{}: rejecting unsupported request {}", (Object)this.persistenceId, request);
            throw new UnsupportedRequestException(request);
        }
        finally {
            this.expectNextRequest();
        }
    }

    private LocalHistorySuccess handleCreateHistory(CreateLocalHistoryRequest request, RequestEnvelope envelope, long now) throws RequestException {
        LocalHistoryIdentifier historyId = (LocalHistoryIdentifier)request.getTarget();
        AbstractFrontendHistory existing = this.localHistories.get(historyId);
        if (existing != null) {
            LOG.debug("{}: history {} already exists", (Object)this.persistenceId, (Object)historyId);
            return new LocalHistorySuccess(historyId, request.getSequence());
        }
        if (this.purgedHistories.contains(historyId.getHistoryId())) {
            LOG.debug("{}: rejecting purged request {}", (Object)this.persistenceId, (Object)request);
            throw new DeadHistoryException(this.purgedHistories.toImmutable());
        }
        if (this.lastSeenHistory == null || Long.compareUnsigned(this.lastSeenHistory, historyId.getHistoryId()) < 0) {
            this.lastSeenHistory = historyId.getHistoryId();
        }
        ShardDataTreeTransactionChain chain = this.tree.ensureTransactionChain(historyId, () -> {
            LOG.debug("{}: persisted history {}", (Object)this.persistenceId, (Object)historyId);
            envelope.sendSuccess((RequestSuccess)new LocalHistorySuccess(historyId, request.getSequence()), this.tree.readTime() - now);
        });
        this.localHistories.put(historyId, LocalFrontendHistory.create(this.persistenceId, this.tree, chain));
        LOG.debug("{}: created history {}", (Object)this.persistenceId, (Object)historyId);
        return null;
    }

    private LocalHistorySuccess handleDestroyHistory(DestroyLocalHistoryRequest request, RequestEnvelope envelope, long now) throws RequestException {
        LocalHistoryIdentifier id = (LocalHistoryIdentifier)request.getTarget();
        LocalFrontendHistory existing = this.localHistories.get(id);
        if (existing == null) {
            LOG.debug("{}: history {} does not exist, nothing to destroy", (Object)this.persistenceId, (Object)id);
            return new LocalHistorySuccess(id, request.getSequence());
        }
        existing.destroy(request.getSequence(), envelope, now);
        return null;
    }

    private LocalHistorySuccess handlePurgeHistory(PurgeLocalHistoryRequest request, RequestEnvelope envelope, long now) throws RequestException {
        LocalHistoryIdentifier id = (LocalHistoryIdentifier)request.getTarget();
        LocalFrontendHistory existing = this.localHistories.remove(id);
        if (existing == null) {
            LOG.debug("{}: history {} has already been purged", (Object)this.persistenceId, (Object)id);
            return new LocalHistorySuccess(id, request.getSequence());
        }
        LOG.debug("{}: purging history {}", (Object)this.persistenceId, (Object)id);
        this.purgedHistories.add(id.getHistoryId());
        existing.purge(request.getSequence(), envelope, now);
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    TransactionSuccess<?> handleTransactionRequest(TransactionRequest<?> request, RequestEnvelope envelope, long now) throws RequestException {
        this.checkRequestSequence(envelope);
        try {
            AbstractFrontendHistory history;
            LocalHistoryIdentifier lhId = ((TransactionIdentifier)request.getTarget()).getHistoryId();
            if (lhId.getHistoryId() != 0L) {
                history = this.localHistories.get(lhId);
                if (history == null) {
                    if (this.purgedHistories.contains(lhId.getHistoryId())) {
                        LOG.warn("{}: rejecting request {} to purged history", (Object)this.persistenceId, request);
                        throw new DeadHistoryException(this.purgedHistories.toImmutable());
                    }
                    LOG.warn("{}: rejecting unknown history request {}", (Object)this.persistenceId, request);
                    throw new UnknownHistoryException(this.lastSeenHistory);
                }
            } else {
                history = this.standaloneHistory;
            }
            TransactionSuccess<?> transactionSuccess = history.handleTransactionRequest(request, envelope, now);
            return transactionSuccess;
        }
        finally {
            this.expectNextRequest();
        }
    }

    void reconnect() {
        this.expectedTxSequence = 0L;
        this.lastConnectTicks = this.tree.readTime();
    }

    void retire() {
        Iterator<SimpleShardDataTreeCohort> it = this.tree.cohortIterator();
        while (it.hasNext()) {
            SimpleShardDataTreeCohort cohort = it.next();
            if (!this.clientId.equals((Object)cohort.getIdentifier().getHistoryId().getClientId())) continue;
            if (cohort.getState() != ShardDataTreeCohort.State.COMMIT_PENDING) {
                LOG.debug("{}: Retiring transaction {}", (Object)this.persistenceId, (Object)cohort.getIdentifier());
                it.remove();
                continue;
            }
            LOG.debug("{}: Transaction {} already committing, not retiring it", (Object)this.persistenceId, (Object)cohort.getIdentifier());
        }
        this.localHistories.values().forEach(AbstractFrontendHistory::retire);
        this.localHistories.clear();
        this.standaloneHistory.retire();
    }

    long getLastConnectTicks() {
        return this.lastConnectTicks;
    }

    long getLastSeenTicks() {
        return this.lastSeenTicks;
    }

    void touch() {
        this.lastSeenTicks = this.tree.readTime();
    }

    public String toString() {
        return MoreObjects.toStringHelper(LeaderFrontendState.class).add("clientId", (Object)this.clientId).add("nanosAgo", this.tree.readTime() - this.lastSeenTicks).add("purgedHistories", (Object)this.purgedHistories).toString();
    }
}

