/*
 * Copyright (c) 2016 Cisco Systems, Inc. and others.  All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 */
package org.opendaylight.controller.cluster.databroker.actors.dds;

import static akka.pattern.Patterns.ask;
import static com.google.common.base.Verify.verifyNotNull;

import akka.dispatch.ExecutionContexts;
import akka.dispatch.OnComplete;
import akka.util.Timeout;
import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableBiMap;
import com.google.common.collect.ImmutableBiMap.Builder;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import org.opendaylight.controller.cluster.access.client.BackendInfoResolver;
import org.opendaylight.controller.cluster.access.concepts.ClientIdentifier;
import org.opendaylight.controller.cluster.datastore.shardmanager.RegisterForShardAvailabilityChanges;
import org.opendaylight.controller.cluster.datastore.shardstrategy.DefaultShardStrategy;
import org.opendaylight.controller.cluster.datastore.utils.ActorContext;
import org.opendaylight.yangtools.concepts.Registration;
import org.opendaylight.yangtools.yang.data.api.YangInstanceIdentifier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import scala.concurrent.Future;

/**
 * {@link BackendInfoResolver} implementation for static shard configuration based on ShardManager. Each string-named
 * shard is assigned a single cookie and this mapping is stored in a bidirectional map. Information about corresponding
 * shard leader is resolved via {@link ActorContext}. The product of resolution is {@link ShardBackendInfo}.
 *
 * @author Robert Varga
 */
@ThreadSafe
final class ModuleShardBackendResolver extends AbstractShardBackendResolver {
    private static final Logger LOG = LoggerFactory.getLogger(ModuleShardBackendResolver.class);

    private final ConcurrentMap<Long, ShardState> backends = new ConcurrentHashMap<>();

    private final Future<Registration> shardAvailabilityChangesRegFuture;

    @GuardedBy("this")
    private long nextShard = 1;

    private volatile BiMap<String, Long> shards = ImmutableBiMap.of(DefaultShardStrategy.DEFAULT_SHARD, 0L);

    // FIXME: we really need just ActorContext.findPrimaryShardAsync()
    ModuleShardBackendResolver(final ClientIdentifier clientId, final ActorContext actorContext) {
        super(clientId, actorContext);

        shardAvailabilityChangesRegFuture = ask(actorContext.getShardManager(), new RegisterForShardAvailabilityChanges(
            this::onShardAvailabilityChange), Timeout.apply(60, TimeUnit.MINUTES))
                .map(reply -> (Registration)reply, ExecutionContexts.global());

        shardAvailabilityChangesRegFuture.onComplete(new OnComplete<Registration>() {
            @Override
            public void onComplete(Throwable failure, Registration reply) {
                if (failure != null) {
                    LOG.error("RegisterForShardAvailabilityChanges failed", failure);
                }
            }
        }, ExecutionContexts.global());
    }

    private void onShardAvailabilityChange(String shardName) {
        LOG.debug("onShardAvailabilityChange for {}", shardName);

        Long cookie = shards.get(shardName);
        if (cookie == null) {
            LOG.debug("No shard cookie found for {}", shardName);
            return;
        }

        notifyStaleBackendInfoCallbacks(cookie);
    }

    Long resolveShardForPath(final YangInstanceIdentifier path) {
        final String shardName = actorContext().getShardStrategyFactory().getStrategy(path).findShard(path);
        Long cookie = shards.get(shardName);
        if (cookie == null) {
            synchronized (this) {
                cookie = shards.get(shardName);
                if (cookie == null) {
                    cookie = nextShard++;

                    Builder<String, Long> builder = ImmutableBiMap.builder();
                    builder.putAll(shards);
                    builder.put(shardName, cookie);
                    shards = builder.build();
                }
            }
        }

        return cookie;
    }

    @Override
    public CompletionStage<ShardBackendInfo> getBackendInfo(final Long cookie) {
        /*
         * We cannot perform a simple computeIfAbsent() here because we need to control sequencing of when the state
         * is inserted into the map and retired from it (based on the stage result).
         *
         * We do not want to hook another stage one processing completes and hooking a removal on failure from a compute
         * method runs the inherent risk of stage completing before the insertion does (i.e. we have a removal of
         * non-existent element.
         */
        final ShardState existing = backends.get(cookie);
        if (existing != null) {
            return existing.getStage();
        }

        final String shardName = shards.inverse().get(cookie);
        if (shardName == null) {
            LOG.warn("Failing request for non-existent cookie {}", cookie);
            throw new IllegalArgumentException("Cookie " + cookie + " does not have a shard assigned");
        }

        LOG.debug("Resolving cookie {} to shard {}", cookie, shardName);
        final ShardState toInsert = resolveBackendInfo(shardName, cookie);

        final ShardState raced = backends.putIfAbsent(cookie, toInsert);
        if (raced != null) {
            // We have had a concurrent insertion, return that
            LOG.debug("Race during insertion of state for cookie {} shard {}", cookie, shardName);
            return raced.getStage();
        }

        // We have succeeded in populating the map, now we need to take care of pruning the entry if it fails to
        // complete
        final CompletionStage<ShardBackendInfo> stage = toInsert.getStage();
        stage.whenComplete((info, failure) -> {
            if (failure != null) {
                LOG.debug("Resolution of cookie {} shard {} failed, removing state", cookie, shardName, failure);
                backends.remove(cookie, toInsert);

                // Remove cache state in case someone else forgot to invalidate it
                flushCache(shardName);
            }
        });

        return stage;
    }

    @Override
    public CompletionStage<ShardBackendInfo> refreshBackendInfo(final Long cookie,
            final ShardBackendInfo staleInfo) {
        final ShardState existing = backends.get(cookie);
        if (existing != null) {
            if (!staleInfo.equals(existing.getResult())) {
                return existing.getStage();
            }

            LOG.debug("Invalidating backend information {}", staleInfo);
            flushCache(staleInfo.getName());

            LOG.trace("Invalidated cache {}", staleInfo);
            backends.remove(cookie, existing);
        }

        return getBackendInfo(cookie);
    }

    @Override
    public void close() {
        shardAvailabilityChangesRegFuture.onComplete(new OnComplete<Registration>() {
            @Override
            public void onComplete(Throwable failure, Registration reply) {
                reply.close();
            }
        }, ExecutionContexts.global());
    }

    @Override
    public String resolveCookieName(Long cookie) {
        return verifyNotNull(shards.inverse().get(cookie), "Unexpected null cookie: %s", cookie);
    }
}
