package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.controller.md.sal.clustering.entity.owners.rev150804.entity.owners;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.controller.md.sal.clustering.entity.owners.rev150804.entity.owners.entity.type.Entity;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link EntityTypeBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     EntityTypeBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new EntityTypeBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of EntityTypeBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see EntityTypeBuilder
 * @see Builder
 *
 */
public class EntityTypeBuilder implements Builder<EntityType> {

    private List<Entity> _entity;
    private String _type;
    private EntityTypeKey key;


    Map<Class<? extends Augmentation<EntityType>>, Augmentation<EntityType>> augmentation = Collections.emptyMap();

    public EntityTypeBuilder() {
    }

    public EntityTypeBuilder(EntityType base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<EntityType>>, Augmentation<EntityType>> aug =((AugmentationHolder<EntityType>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this.key = base.key();
        this._type = base.getType();
        this._entity = base.getEntity();
    }


    public EntityTypeKey key() {
        return key;
    }
    
    public List<Entity> getEntity() {
        return _entity;
    }
    
    public String getType() {
        return _type;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<EntityType>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    public EntityTypeBuilder withKey(final EntityTypeKey key) {
        this.key = key;
        return this;
    }
    public EntityTypeBuilder setEntity(final List<Entity> values) {
        this._entity = values;
        return this;
    }
    
    
    public EntityTypeBuilder setType(final String value) {
        this._type = value;
        return this;
    }
    
    public EntityTypeBuilder addAugmentation(Class<? extends Augmentation<EntityType>> augmentationType, Augmentation<EntityType> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public EntityTypeBuilder removeAugmentation(Class<? extends Augmentation<EntityType>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public EntityType build() {
        return new EntityTypeImpl(this);
    }

    private static final class EntityTypeImpl
        extends AbstractAugmentable<EntityType>
        implements EntityType {
    
        private final List<Entity> _entity;
        private final String _type;
        private final EntityTypeKey key;
    
        EntityTypeImpl(EntityTypeBuilder base) {
            super(base.augmentation);
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new EntityTypeKey(base.getType());
            }
            this._type = key.getType();
            this._entity = base.getEntity();
        }
    
        @Override
        public EntityTypeKey key() {
            return key;
        }
        
        @Override
        public List<Entity> getEntity() {
            return _entity;
        }
        
        @Override
        public String getType() {
            return _type;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_entity);
            result = prime * result + Objects.hashCode(_type);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!EntityType.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            EntityType other = (EntityType)obj;
            if (!Objects.equals(_entity, other.getEntity())) {
                return false;
            }
            if (!Objects.equals(_type, other.getType())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                EntityTypeImpl otherImpl = (EntityTypeImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<EntityType>>, Augmentation<EntityType>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("EntityType");
            CodeHelpers.appendValue(helper, "_entity", _entity);
            CodeHelpers.appendValue(helper, "_type", _type);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
