package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.controller.md.sal.clustering.entity.owners.rev150804.entity.owners.entity.type;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.controller.md.sal.clustering.entity.owners.rev150804.entity.owners.entity.type.entity.Candidate;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;
import org.opendaylight.yangtools.yang.binding.InstanceIdentifier;

/**
 * Class that builds {@link EntityBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     EntityBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new EntityBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of EntityBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see EntityBuilder
 * @see Builder
 *
 */
public class EntityBuilder implements Builder<Entity> {

    private List<Candidate> _candidate;
    private InstanceIdentifier<?> _id;
    private String _owner;
    private EntityKey key;


    Map<Class<? extends Augmentation<Entity>>, Augmentation<Entity>> augmentation = Collections.emptyMap();

    public EntityBuilder() {
    }

    public EntityBuilder(Entity base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<Entity>>, Augmentation<Entity>> aug =((AugmentationHolder<Entity>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this.key = base.key();
        this._id = base.getId();
        this._candidate = base.getCandidate();
        this._owner = base.getOwner();
    }


    public EntityKey key() {
        return key;
    }
    
    public List<Candidate> getCandidate() {
        return _candidate;
    }
    
    public InstanceIdentifier<?> getId() {
        return _id;
    }
    
    public String getOwner() {
        return _owner;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<Entity>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    public EntityBuilder withKey(final EntityKey key) {
        this.key = key;
        return this;
    }
    public EntityBuilder setCandidate(final List<Candidate> values) {
        this._candidate = values;
        return this;
    }
    
    
    public EntityBuilder setId(final InstanceIdentifier<?> value) {
        this._id = value;
        return this;
    }
    
    public EntityBuilder setOwner(final String value) {
        this._owner = value;
        return this;
    }
    
    public EntityBuilder addAugmentation(Class<? extends Augmentation<Entity>> augmentationType, Augmentation<Entity> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public EntityBuilder removeAugmentation(Class<? extends Augmentation<Entity>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public Entity build() {
        return new EntityImpl(this);
    }

    private static final class EntityImpl
        extends AbstractAugmentable<Entity>
        implements Entity {
    
        private final List<Candidate> _candidate;
        private final InstanceIdentifier<?> _id;
        private final String _owner;
        private final EntityKey key;
    
        EntityImpl(EntityBuilder base) {
            super(base.augmentation);
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new EntityKey(base.getId());
            }
            this._id = key.getId();
            this._candidate = base.getCandidate();
            this._owner = base.getOwner();
        }
    
        @Override
        public EntityKey key() {
            return key;
        }
        
        @Override
        public List<Candidate> getCandidate() {
            return _candidate;
        }
        
        @Override
        public InstanceIdentifier<?> getId() {
            return _id;
        }
        
        @Override
        public String getOwner() {
            return _owner;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_candidate);
            result = prime * result + Objects.hashCode(_id);
            result = prime * result + Objects.hashCode(_owner);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!Entity.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            Entity other = (Entity)obj;
            if (!Objects.equals(_candidate, other.getCandidate())) {
                return false;
            }
            if (!Objects.equals(_id, other.getId())) {
                return false;
            }
            if (!Objects.equals(_owner, other.getOwner())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                EntityImpl otherImpl = (EntityImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<Entity>>, Augmentation<Entity>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("Entity");
            CodeHelpers.appendValue(helper, "_candidate", _candidate);
            CodeHelpers.appendValue(helper, "_id", _id);
            CodeHelpers.appendValue(helper, "_owner", _owner);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
