/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.controller.md.sal.dom.store.impl;

import com.google.common.annotations.Beta;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import java.util.Collection;
import java.util.Map;
import org.opendaylight.controller.md.sal.common.api.data.AsyncDataBroker;
import org.opendaylight.controller.md.sal.dom.spi.RegistrationTreeSnapshot;
import org.opendaylight.controller.md.sal.dom.store.impl.DOMImmutableDataChangeEvent;
import org.opendaylight.controller.md.sal.dom.store.impl.DataChangeListenerRegistration;
import org.opendaylight.controller.md.sal.dom.store.impl.ResolveDataChangeState;
import org.opendaylight.controller.md.sal.dom.store.impl.tree.ListenerTree;
import org.opendaylight.yangtools.util.concurrent.NotificationManager;
import org.opendaylight.yangtools.yang.data.api.YangInstanceIdentifier;
import org.opendaylight.yangtools.yang.data.api.schema.NormalizedNode;
import org.opendaylight.yangtools.yang.data.api.schema.NormalizedNodeContainer;
import org.opendaylight.yangtools.yang.data.api.schema.tree.DataTreeCandidate;
import org.opendaylight.yangtools.yang.data.api.schema.tree.DataTreeCandidateNode;
import org.opendaylight.yangtools.yang.data.api.schema.tree.ModificationType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Beta
public final class ResolveDataChangeEventsTask {
    private static final Logger LOG = LoggerFactory.getLogger(ResolveDataChangeEventsTask.class);
    private final DataTreeCandidate candidate;
    private final ListenerTree listenerRoot;
    private Multimap<DataChangeListenerRegistration<?>, DOMImmutableDataChangeEvent> collectedEvents;

    private ResolveDataChangeEventsTask(DataTreeCandidate candidate, ListenerTree listenerTree) {
        this.candidate = (DataTreeCandidate)Preconditions.checkNotNull((Object)candidate);
        this.listenerRoot = (ListenerTree)((Object)Preconditions.checkNotNull((Object)((Object)listenerTree)));
    }

    public synchronized void resolve(NotificationManager<DataChangeListenerRegistration<?>, DOMImmutableDataChangeEvent> manager) {
        try (RegistrationTreeSnapshot w = this.listenerRoot.takeSnapshot();){
            this.collectedEvents = ArrayListMultimap.create();
            ResolveDataChangeState s = ResolveDataChangeState.initial(this.candidate.getRootPath(), w.getRootNode());
            this.resolveAnyChangeEvent(s, this.candidate.getRootNode());
            for (Map.Entry e : this.collectedEvents.asMap().entrySet()) {
                DOMImmutableDataChangeEvent event;
                Collection col = (Collection)e.getValue();
                if (col.size() != 1) {
                    DOMImmutableDataChangeEvent.Builder b = DOMImmutableDataChangeEvent.builder(AsyncDataBroker.DataChangeScope.BASE);
                    for (DOMImmutableDataChangeEvent i : col) {
                        b.merge(i);
                    }
                    event = b.build();
                    LOG.trace("Merged events {} into event {}", (Object)col, (Object)event);
                } else {
                    event = (DOMImmutableDataChangeEvent)col.iterator().next();
                }
                manager.submitNotification(e.getKey(), (Object)event);
            }
        }
    }

    private boolean resolveAnyChangeEvent(ResolveDataChangeState state, DataTreeCandidateNode node) {
        Optional maybeBefore = node.getDataBefore();
        Optional maybeAfter = node.getDataAfter();
        ModificationType type = node.getModificationType();
        if (type != ModificationType.UNMODIFIED && !maybeAfter.isPresent() && !maybeBefore.isPresent()) {
            LOG.debug("Modification at {} has type {}, but no before- and after-data. Assuming unchanged.", (Object)state.getPath(), (Object)type);
            return false;
        }
        switch (type) {
            case SUBTREE_MODIFIED: {
                return this.resolveSubtreeChangeEvent(state, node);
            }
            case APPEARED: 
            case WRITE: {
                Preconditions.checkArgument((boolean)maybeAfter.isPresent(), (String)"Modification at {} has type {} but no after-data", (Object[])new Object[]{state.getPath(), type});
                if (!maybeBefore.isPresent()) {
                    NormalizedNode afterNode = (NormalizedNode)maybeAfter.get();
                    this.resolveSameEventRecursivelly(state, afterNode, DOMImmutableDataChangeEvent.getCreateEventFactory());
                    return true;
                }
                return this.resolveReplacedEvent(state, (NormalizedNode)maybeBefore.get(), (NormalizedNode)maybeAfter.get());
            }
            case DISAPPEARED: 
            case DELETE: {
                Preconditions.checkArgument((boolean)maybeBefore.isPresent(), (String)"Modification at {} has type {} but no before-data", (Object[])new Object[]{state.getPath(), type});
                NormalizedNode beforeNode = (NormalizedNode)maybeBefore.get();
                this.resolveSameEventRecursivelly(state, beforeNode, DOMImmutableDataChangeEvent.getRemoveEventFactory());
                return true;
            }
            case UNMODIFIED: {
                return false;
            }
        }
        throw new IllegalStateException(String.format("Unhandled node state %s at %s", type, state.getPath()));
    }

    private boolean resolveReplacedEvent(ResolveDataChangeState state, NormalizedNode<?, ?> beforeData, NormalizedNode<?, ?> afterData) {
        if (beforeData instanceof NormalizedNodeContainer) {
            LOG.trace("Resolving subtree replace event for {} before {}, after {}", new Object[]{state.getPath(), beforeData, afterData});
            NormalizedNodeContainer beforeCont = (NormalizedNodeContainer)beforeData;
            NormalizedNodeContainer afterCont = (NormalizedNodeContainer)afterData;
            return this.resolveNodeContainerReplaced(state, beforeCont, afterCont);
        }
        if (beforeData.equals(afterData)) {
            LOG.trace("Skipping equal leaf {}", (Object)state.getPath());
            return false;
        }
        LOG.trace("Resolving leaf replace event for {} , before {}, after {}", new Object[]{state.getPath(), beforeData, afterData});
        DOMImmutableDataChangeEvent event = DOMImmutableDataChangeEvent.builder(AsyncDataBroker.DataChangeScope.BASE).addUpdated(state.getPath(), beforeData, afterData).build();
        state.addEvent(event);
        state.collectEvents(beforeData, afterData, this.collectedEvents);
        return true;
    }

    private boolean resolveNodeContainerReplaced(ResolveDataChangeState state, NormalizedNodeContainer<?, YangInstanceIdentifier.PathArgument, NormalizedNode<YangInstanceIdentifier.PathArgument, ?>> beforeCont, NormalizedNodeContainer<?, YangInstanceIdentifier.PathArgument, NormalizedNode<YangInstanceIdentifier.PathArgument, ?>> afterCont) {
        YangInstanceIdentifier.PathArgument childId;
        if (!state.needsProcessing()) {
            LOG.trace("Not processing replaced container {}", (Object)state.getPath());
            return true;
        }
        boolean childChanged = false;
        for (NormalizedNode beforeChild : beforeCont.getValue()) {
            childId = beforeChild.getIdentifier();
            if (!this.resolveNodeContainerChildUpdated(state.child(childId), beforeChild, afterCont.getChild(childId))) continue;
            childChanged = true;
        }
        for (NormalizedNode afterChild : afterCont.getValue()) {
            childId = afterChild.getIdentifier();
            if (beforeCont.getChild(childId).isPresent()) continue;
            this.resolveSameEventRecursivelly(state.child(childId), afterChild, DOMImmutableDataChangeEvent.getCreateEventFactory());
            childChanged = true;
        }
        if (childChanged) {
            DOMImmutableDataChangeEvent event = DOMImmutableDataChangeEvent.builder(AsyncDataBroker.DataChangeScope.BASE).addUpdated(state.getPath(), (NormalizedNode<?, ?>)beforeCont, (NormalizedNode<?, ?>)afterCont).build();
            state.addEvent(event);
        }
        state.collectEvents((NormalizedNode<?, ?>)beforeCont, (NormalizedNode<?, ?>)afterCont, this.collectedEvents);
        return childChanged;
    }

    private boolean resolveNodeContainerChildUpdated(ResolveDataChangeState state, NormalizedNode<YangInstanceIdentifier.PathArgument, ?> before, Optional<NormalizedNode<YangInstanceIdentifier.PathArgument, ?>> after) {
        if (after.isPresent()) {
            return this.resolveReplacedEvent(state, before, (NormalizedNode)after.get());
        }
        this.resolveSameEventRecursivelly(state, before, DOMImmutableDataChangeEvent.getRemoveEventFactory());
        return true;
    }

    private void resolveSameEventRecursivelly(ResolveDataChangeState state, NormalizedNode<YangInstanceIdentifier.PathArgument, ?> node, DOMImmutableDataChangeEvent.SimpleEventFactory eventFactory) {
        if (!state.needsProcessing()) {
            LOG.trace("Skipping child {}", (Object)state.getPath());
            return;
        }
        if (node instanceof NormalizedNodeContainer) {
            LOG.trace("Resolving subtree recursive event for {}, type {}", (Object)state.getPath(), (Object)eventFactory);
            NormalizedNodeContainer container = (NormalizedNodeContainer)node;
            for (NormalizedNode child : container.getValue()) {
                YangInstanceIdentifier.PathArgument childId = child.getIdentifier();
                LOG.trace("Resolving event for child {}", (Object)childId);
                this.resolveSameEventRecursivelly(state.child(childId), child, eventFactory);
            }
        }
        DOMImmutableDataChangeEvent event = eventFactory.create(state.getPath(), node);
        LOG.trace("Adding event {} at path {}", (Object)event, (Object)state.getPath());
        state.addEvent(event);
        state.collectEvents(event.getOriginalSubtree(), event.getUpdatedSubtree(), this.collectedEvents);
    }

    private boolean resolveSubtreeChangeEvent(ResolveDataChangeState state, DataTreeCandidateNode modification) {
        Optional maybeBefore = modification.getDataBefore();
        Optional maybeAfter = modification.getDataAfter();
        Preconditions.checkArgument((boolean)maybeBefore.isPresent(), (String)"Subtree change with before-data not present at path %s", (Object[])new Object[]{state.getPath()});
        Preconditions.checkArgument((boolean)maybeAfter.isPresent(), (String)"Subtree change with after-data not present at path %s", (Object[])new Object[]{state.getPath()});
        if (!state.needsProcessing()) {
            LOG.trace("Not processing modified subtree {}", (Object)state.getPath());
            return true;
        }
        AsyncDataBroker.DataChangeScope scope = null;
        for (DataTreeCandidateNode childMod : modification.getChildNodes()) {
            ResolveDataChangeState childState = state.child(childMod.getIdentifier());
            switch (childMod.getModificationType()) {
                case APPEARED: 
                case WRITE: 
                case DISAPPEARED: 
                case DELETE: {
                    if (!this.resolveAnyChangeEvent(childState, childMod)) break;
                    scope = AsyncDataBroker.DataChangeScope.ONE;
                    break;
                }
                case SUBTREE_MODIFIED: {
                    if (!this.resolveSubtreeChangeEvent(childState, childMod) || scope != null) break;
                    scope = AsyncDataBroker.DataChangeScope.SUBTREE;
                    break;
                }
            }
        }
        NormalizedNode before = (NormalizedNode)maybeBefore.get();
        NormalizedNode after = (NormalizedNode)maybeAfter.get();
        if (scope != null) {
            DOMImmutableDataChangeEvent one = DOMImmutableDataChangeEvent.builder(scope).addUpdated(state.getPath(), before, after).build();
            state.addEvent(one);
        }
        state.collectEvents(before, after, this.collectedEvents);
        return scope != null;
    }

    public static ResolveDataChangeEventsTask create(DataTreeCandidate candidate, ListenerTree listenerTree) {
        return new ResolveDataChangeEventsTask(candidate, listenerTree);
    }
}

