package org.opendaylight.yang.gen.v1.http.netconfcentral.org.ns.toaster.rev091120;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Long;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link MakeToastInputBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     MakeToastInputBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new MakeToastInputBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of MakeToastInputBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see MakeToastInputBuilder
 * @see Builder
 *
 */
public class MakeToastInputBuilder implements Builder<MakeToastInput> {

    private Long _toasterDoneness;
    private Class<? extends ToastType> _toasterToastType;


    Map<Class<? extends Augmentation<MakeToastInput>>, Augmentation<MakeToastInput>> augmentation = Collections.emptyMap();

    public MakeToastInputBuilder() {
    }

    public MakeToastInputBuilder(MakeToastInput base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<MakeToastInput>>, Augmentation<MakeToastInput>> aug =((AugmentationHolder<MakeToastInput>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._toasterDoneness = base.getToasterDoneness();
        this._toasterToastType = base.getToasterToastType();
    }


    public Long getToasterDoneness() {
        return _toasterDoneness;
    }
    
    public Class<? extends ToastType> getToasterToastType() {
        return _toasterToastType;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<MakeToastInput>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    private static void checkToasterDonenessRange(final long value) {
        if (value >= 1L && value <= 10L) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[1..10]]", value);
    }
    
    public MakeToastInputBuilder setToasterDoneness(final Long value) {
    if (value != null) {
        checkToasterDonenessRange(value);
        
    }
        this._toasterDoneness = value;
        return this;
    }
    
    public MakeToastInputBuilder setToasterToastType(final Class<? extends ToastType> value) {
        this._toasterToastType = value;
        return this;
    }
    
    public MakeToastInputBuilder addAugmentation(Class<? extends Augmentation<MakeToastInput>> augmentationType, Augmentation<MakeToastInput> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public MakeToastInputBuilder removeAugmentation(Class<? extends Augmentation<MakeToastInput>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public MakeToastInput build() {
        return new MakeToastInputImpl(this);
    }

    private static final class MakeToastInputImpl
        extends AbstractAugmentable<MakeToastInput>
        implements MakeToastInput {
    
        private final Long _toasterDoneness;
        private final Class<? extends ToastType> _toasterToastType;
    
        MakeToastInputImpl(MakeToastInputBuilder base) {
            super(base.augmentation);
            this._toasterDoneness = base.getToasterDoneness();
            this._toasterToastType = base.getToasterToastType();
        }
    
        @Override
        public Long getToasterDoneness() {
            return _toasterDoneness;
        }
        
        @Override
        public Class<? extends ToastType> getToasterToastType() {
            return _toasterToastType;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_toasterDoneness);
            result = prime * result + Objects.hashCode(_toasterToastType);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!MakeToastInput.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            MakeToastInput other = (MakeToastInput)obj;
            if (!Objects.equals(_toasterDoneness, other.getToasterDoneness())) {
                return false;
            }
            if (!Objects.equals(_toasterToastType, other.getToasterToastType())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                MakeToastInputImpl otherImpl = (MakeToastInputImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<MakeToastInput>>, Augmentation<MakeToastInput>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("MakeToastInput");
            CodeHelpers.appendValue(helper, "_toasterDoneness", _toasterDoneness);
            CodeHelpers.appendValue(helper, "_toasterToastType", _toasterToastType);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
