package org.opendaylight.yang.gen.v1.urn.detnet.bandwidth.api.rev180907;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Long;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.detnet.bandwidth.api.rev180907.link.bandwidth.change.OldLink;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;
import org.opendaylight.yangtools.yang.common.Uint32;

/**
 * Class that builds {@link LinkBandwidthChangeBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     LinkBandwidthChangeBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new LinkBandwidthChangeBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of LinkBandwidthChangeBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see LinkBandwidthChangeBuilder
 * @see Builder
 *
 */
public class LinkBandwidthChangeBuilder implements Builder<LinkBandwidthChange> {

    private Uint32 _newAvailableBandwidth;
    private OldLink _oldLink;


    Map<Class<? extends Augmentation<LinkBandwidthChange>>, Augmentation<LinkBandwidthChange>> augmentation = Collections.emptyMap();

    public LinkBandwidthChangeBuilder() {
    }

    public LinkBandwidthChangeBuilder(LinkBandwidthChange base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<LinkBandwidthChange>>, Augmentation<LinkBandwidthChange>> aug =((AugmentationHolder<LinkBandwidthChange>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._newAvailableBandwidth = base.getNewAvailableBandwidth();
        this._oldLink = base.getOldLink();
    }


    public Uint32 getNewAvailableBandwidth() {
        return _newAvailableBandwidth;
    }
    
    public OldLink getOldLink() {
        return _oldLink;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<LinkBandwidthChange>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    
    public LinkBandwidthChangeBuilder setNewAvailableBandwidth(final Uint32 value) {
        this._newAvailableBandwidth = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setNewAvailableBandwidth(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public LinkBandwidthChangeBuilder setNewAvailableBandwidth(final Long value) {
        return setNewAvailableBandwidth(CodeHelpers.compatUint(value));
    }
    
    public LinkBandwidthChangeBuilder setOldLink(final OldLink value) {
        this._oldLink = value;
        return this;
    }
    
    public LinkBandwidthChangeBuilder addAugmentation(Class<? extends Augmentation<LinkBandwidthChange>> augmentationType, Augmentation<LinkBandwidthChange> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public LinkBandwidthChangeBuilder removeAugmentation(Class<? extends Augmentation<LinkBandwidthChange>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public LinkBandwidthChange build() {
        return new LinkBandwidthChangeImpl(this);
    }

    private static final class LinkBandwidthChangeImpl
        extends AbstractAugmentable<LinkBandwidthChange>
        implements LinkBandwidthChange {
    
        private final Uint32 _newAvailableBandwidth;
        private final OldLink _oldLink;
    
        LinkBandwidthChangeImpl(LinkBandwidthChangeBuilder base) {
            super(base.augmentation);
            this._newAvailableBandwidth = base.getNewAvailableBandwidth();
            this._oldLink = base.getOldLink();
        }
    
        @Override
        public Uint32 getNewAvailableBandwidth() {
            return _newAvailableBandwidth;
        }
        
        @Override
        public OldLink getOldLink() {
            return _oldLink;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_newAvailableBandwidth);
            result = prime * result + Objects.hashCode(_oldLink);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!LinkBandwidthChange.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            LinkBandwidthChange other = (LinkBandwidthChange)obj;
            if (!Objects.equals(_newAvailableBandwidth, other.getNewAvailableBandwidth())) {
                return false;
            }
            if (!Objects.equals(_oldLink, other.getOldLink())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                LinkBandwidthChangeImpl otherImpl = (LinkBandwidthChangeImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<LinkBandwidthChange>>, Augmentation<LinkBandwidthChange>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("LinkBandwidthChange");
            CodeHelpers.appendValue(helper, "_newAvailableBandwidth", _newAvailableBandwidth);
            CodeHelpers.appendValue(helper, "_oldLink", _oldLink);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
